<?php
/**
 * Generate xPDO model classes from schema (MODX 3 / xPDO 3).
 *
 * Run from site root:
 *   php core/components/i18n/_build/build.model.php
 */

declare(strict_types=1);

use MODX\Revolution\modX;

// --- Find site root (where config.core.php is)
$root = __DIR__;
while (!file_exists($root . '/config.core.php')) {
    $parent = dirname($root);
    if ($parent === $root) {
        die("Could not locate config.core.php\n");
    }
    $root = $parent;
}

require_once $root . '/config.core.php';
require_once MODX_CORE_PATH . 'vendor/autoload.php';

/** @var modX $modx */
$modx = new modX();
$modx->initialize('mgr');

$modx->setLogLevel(modX::LOG_LEVEL_INFO);
$modx->setLogTarget('ECHO');

$component = 'i18n';
$platform  = 'mysql';

$corePath    = $modx->getOption('core_path') . "components/{$component}/";
$schemaFile  = $corePath . "model/schema/{$component}.{$platform}.schema.xml";

$modelParent = $corePath . "model/";
$pkgDir      = $modelParent . "{$component}/";

if (!file_exists($schemaFile)) {
    $modx->log(modX::LOG_LEVEL_ERROR, "[i18n] Schema not found: {$schemaFile}");
    exit(1);
}

if (!is_dir($pkgDir) && !@mkdir($pkgDir, 0777, true) && !is_dir($pkgDir)) {
    $modx->log(modX::LOG_LEVEL_ERROR, "[i18n] Could not create model dir: {$pkgDir}");
    exit(1);
}

$manager   = $modx->getManager();
$generator = $manager ? $manager->getGenerator() : null;

if (!$generator) {
    $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] Could not get xPDO generator.');
    exit(1);
}

$modx->log(modX::LOG_LEVEL_INFO, "[i18n] Generating model from schema:");
$modx->log(modX::LOG_LEVEL_INFO, " - schema: {$schemaFile}");
$modx->log(modX::LOG_LEVEL_INFO, " - output parent: {$modelParent}");

// Cleanup common outputs (safe)
$cleanup = [
    $pkgDir . 'metadata.mysql.php',
    $pkgDir . 'I18nKey.php',
    $pkgDir . 'I18nValue.php',
    $pkgDir . 'I18nUsage.php',
];
foreach ($cleanup as $f) {
    if (file_exists($f)) @unlink($f);
}

$options = [
    'package'     => $component,
    'platform'    => $platform,
    'regenerate'  => true,
    'update'      => false,
];

$called = false;
$lastError = null;

try {
    $generator->parseSchema($schemaFile, $modelParent, $options);
    $called = true;
} catch (Throwable $e) {
    $lastError = $e;
}

if (!$called) {
    // legacy fallback
    try {
        $generator->parseSchema($schemaFile, $modelParent, $component, true, false);
        $called = true;
    } catch (Throwable $e) {
        $lastError = $e;
    }
}

if (!$called) {
    $msg = $lastError ? $lastError->getMessage() : 'Unknown error';
    $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] parseSchema failed: ' . $msg);
    exit(1);
}

// Validate PSR-4 outputs
$expected = [
    $pkgDir . 'metadata.mysql.php',
    $pkgDir . 'I18nKey.php',
    $pkgDir . 'I18nValue.php',
    $pkgDir . 'I18nUsage.php',
];

$missing = [];
foreach ($expected as $f) {
    if (!file_exists($f)) $missing[] = $f;
}

if ($missing) {
    $modx->log(modX::LOG_LEVEL_ERROR, "[i18n] Model generation finished, but some files are missing:");
    foreach ($missing as $m) {
        $modx->log(modX::LOG_LEVEL_ERROR, " - {$m}");
    }
    exit(1);
}

$modx->log(modX::LOG_LEVEL_INFO, '[i18n] Model generation completed successfully.');
exit(0);
