<?php
/**
 * Build i18n transport package (MODX 3).
 *
 * Run from site root:
 *   php core/components/i18n/_build/build.transport.php
 */

declare(strict_types=1);

use MODX\Revolution\modX;
use MODX\Revolution\modNamespace;
use MODX\Revolution\modSystemSetting;
use MODX\Revolution\modMenu;
use MODX\Revolution\modSnippet;
use MODX\Revolution\modCategory;
use MODX\Revolution\modPlugin;
use MODX\Revolution\modPluginEvent;
use MODX\Revolution\Transport\modPackageBuilder;
use xPDO\Transport\xPDOTransport;

// --- Find site root (where config.core.php is)
$root = __DIR__;
while (!file_exists($root . '/config.core.php')) {
    $parent = dirname($root);
    if ($parent === $root) {
        die("Could not locate config.core.php\n");
    }
    $root = $parent;
}

require_once $root . '/config.core.php';
require_once MODX_CORE_PATH . 'vendor/autoload.php';

$config = require __DIR__ . '/config.php';

$mtime = microtime(true);

/** @var modX $modx */
$modx = new modX();
$modx->initialize('mgr');

$modx->setLogLevel(modX::LOG_LEVEL_INFO);
$modx->setLogTarget('ECHO');

$name    = (string)$config['name'];
$version = (string)$config['version'];
$release = (string)$config['release'];
$ns      = (string)$config['namespace'];
$lc      = (string)$config['lowercase'];

/** @var modPackageBuilder $builder */
$builder = new modPackageBuilder($modx);

$builder->createPackage($name, $version, $release);
$builder->registerNamespace($ns, false, true, '{core_path}components/' . $lc . '/');

$normalizePhpElement = static function (string $code): string {
    // Remove UTF-8 BOM if present
    $code = preg_replace('/^\xEF\xBB\xBF/', '', $code) ?? $code;
    $code = ltrim($code);

    // Strip opening tags (<?php, <?) at the beginning
    $code = preg_replace('/^\s*<\?(php)?\s*/i', '', $code) ?? $code;

    // Strip closing tag at the end
    $code = preg_replace('/\s*\?>\s*$/', '', $code) ?? $code;

    return trim($code);
};

// ----------
// Sources
// ----------
$sources = [
    'build'     => __DIR__ . '/',
    // Install-time resolvers live here (must exist on build machine)
    'resolvers' => __DIR__ . '/transport_resolvers/',
    // Elements live inside component
    'elements'  => MODX_CORE_PATH . 'components/' . $lc . '/elements/',
];

// Preflight: required resolver files must exist
$requiredResolvers = [
    'resolve.ping.setting.php',
    'resolve.license.php',
    'resolve.tables.xpdo.php',
	'resolve.plugin_events.php',
	'resolve.cleanup.php',
];

foreach ($requiredResolvers as $file) {
    $p = $sources['resolvers'] . $file;
    if (!file_exists($p)) {
        $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] BUILD STOP: missing resolver file: ' . $p);
        exit(1);
    }
}

$readDoc = static function (string $path) use ($modx): string {
    if (!is_file($path)) {
        $modx->log(modX::LOG_LEVEL_WARN, '[i18n] Build: doc file not found: ' . $path);
        return '';
    }
    $c = file_get_contents($path);
    if ($c === false) {
        $modx->log(modX::LOG_LEVEL_WARN, '[i18n] Build: cannot read doc file: ' . $path);
        return '';
    }
    // Optional: normalize line endings
    return trim(str_replace(["\r\n", "\r"], "\n", $c));
};

// Helper: read element content
$readFile = static function (string $path) use ($modx): string {
    if (!is_file($path)) {
        $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] Build: element file not found: ' . $path);
        return '';
    }
    $content = file_get_contents($path);
    if ($content === false) {
        $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] Build: cannot read element file: ' . $path);
        return '';
    }
    return $content;
};

// ----------
// Namespace vehicle
// ----------
$nsObj = $modx->getObject(modNamespace::class, ['name' => $ns]);
if (!$nsObj) {
    $nsObj = $modx->newObject(modNamespace::class);
    $nsObj->set('name', $ns);
}
$nsObj->set('path', '{core_path}components/' . $lc . '/');
$nsObj->set('assets_path', '{assets_path}components/' . $lc . '/');

$nsVehicle = $builder->createVehicle($nsObj, [
    xPDOTransport::UNIQUE_KEY     => 'name',
    xPDOTransport::PRESERVE_KEYS  => true,
    xPDOTransport::UPDATE_OBJECT  => true,
]);
$builder->putVehicle($nsVehicle);

// ----------
// Guaranteed carrier vehicle: i18n._install_signature
// All file resolvers + PHP resolvers are attached here.
// ----------
$installKey = 'i18n._install_signature';
$installVal = $version . '-' . $release . '-' . time();

$installerSetting = $modx->newObject(modSystemSetting::class);
$installerSetting->fromArray([
    'key'       => $installKey,
    'value'     => $installVal,
    'xtype'     => 'textfield',
    'namespace' => $ns,
    'area'      => 'build',
], '', true, true);

$installerVehicle = $builder->createVehicle($installerSetting, [
    xPDOTransport::UNIQUE_KEY      => 'key',
    xPDOTransport::PRESERVE_KEYS   => true,
    xPDOTransport::UPDATE_OBJECT   => true,
    xPDOTransport::RELATED_OBJECTS => false,
]);

// 1) Core files → core/components/i18n
$installerVehicle->resolve('file', [
    'source' => (string)$config['package_dir'],
    'target' => "return MODX_CORE_PATH . 'components/';",
]);

// 2) Assets files → assets/components/i18n
$installerVehicle->resolve('file', [
    'source' => (string)$config['assets_dir'],
    'target' => "return MODX_ASSETS_PATH . 'components/';",
]);

// 3) PHP resolvers
$installerVehicle->resolve('php', [
    'source' => $sources['resolvers'] . 'resolve.ping.setting.php',
]);
$installerVehicle->resolve('php', [
    'source' => $sources['resolvers'] . 'resolve.license.php',
]);
$installerVehicle->resolve('php', [
    'source' => $sources['resolvers'] . 'resolve.tables.xpdo.php',
]);

$installerVehicle->resolve('php', [
    'source' => $sources['resolvers'] . 'resolve.plugin_events.php',
]);

$installerVehicle->resolve('php', [
    'source' => $sources['resolvers'] . 'resolve.cleanup.php',
]);

$builder->putVehicle($installerVehicle);

// ----------
// System settings
// NOTE: license_state/license_payload must not overwrite on upgrade.
// ----------
$settings = [
    'i18n.default_lang'          => ['value' => 'en', 'xtype' => 'textfield',      'area' => 'general'],
    'i18n.allowed_langs'         => ['value' => 'en', 'xtype' => 'textfield',      'area' => 'general'],
    'i18n.default_prefix'        => ['value' => 'manual', 'xtype' => 'textfield',  'area' => 'general'],
    'i18n.cookie_lifetime_days'  => ['value' => '180', 'xtype' => 'numberfield',   'area' => 'general'],
    'i18n.cookie_name'           => ['value' => 'i18n_mgr', 'xtype' => 'textfield','area' => 'general'],
    'i18n.mgr_tools_enabled'     => ['value' => '1', 'xtype' => 'xcheckbox',       'area' => 'general'],
    'i18n.route_tools_enabled'   => ['value' => '1', 'xtype' => 'xcheckbox',       'area' => 'general'],
    'i18n.track_usage'           => ['value' => '1', 'xtype' => 'xcheckbox',       'area' => 'general'],

    'i18n.license_endpoint' => [
        'value' => 'https://gren.studio/PSp5ACuA6KG4AMXDhxzyoyLyr5JQc1Qp/checkin.php',
        'xtype' => 'textfield',
        'area'  => 'license',
    ],
    'i18n.license_token' => [
        'value' => 'U092G4gYCwReh8uCEjiUaTgqTGWNJvUT',
        'xtype' => 'textfield',
        'area'  => 'license',
    ],
    'i18n.license_server_pubkey_b64' => [
        'value' => 'joYxRnpRAkRXThLNHHsFS543GDwg2CSX3R55LxA2LsM=',
        'xtype' => 'textarea',
        'area'  => 'license',
    ],

    // Must not overwrite on upgrade
    'i18n.license_payload' => ['value' => '', 'xtype' => 'textarea', 'area' => 'license'],
    'i18n.license_state'   => ['value' => '', 'xtype' => 'textarea', 'area' => 'license'],

    'i18n.license_checkin_interval' => ['value' => '0', 'xtype' => 'numberfield', 'area' => 'license'],
    'i18n.license_checkin_min_interval' => ['value' => '0', 'xtype' => 'numberfield', 'area' => 'license'],
];

foreach ($settings as $key => $meta) {
    $s = $modx->newObject(modSystemSetting::class);
    $s->fromArray([
        'key'       => $key,
        'value'     => (string)($meta['value'] ?? ''),
        'xtype'     => (string)($meta['xtype'] ?? 'textfield'),
        'namespace' => $ns,
        'area'      => (string)($meta['area'] ?? 'general'),
    ], '', true, true);

    $update = !in_array($key, ['i18n.license_payload', 'i18n.license_state'], true);

    $vehicle = $builder->createVehicle($s, [
        xPDOTransport::UNIQUE_KEY      => 'key',
        xPDOTransport::PRESERVE_KEYS   => true,
        xPDOTransport::UPDATE_OBJECT   => $update,
        xPDOTransport::RELATED_OBJECTS => false,
    ]);

    $builder->putVehicle($vehicle);
}

// ----------
// Elements: pack snippets + plugins (+ plugin events)
// Convention:
// - snippets: core/components/i18n/elements/snippets/*.php
// - plugins:  core/components/i18n/elements/plugins/*.php
//
// You MUST list which ones to pack here (deterministic builds).
// ----------
$elements = [
    'snippets' => [
         'i18nKey' => 'i18nKey.php',
         'i18nGetLang' => 'i18nGetLang.php',
    ],

    'plugins' => [
         'i18nManagerTools' => [
             'file' => 'i18nManagerTools.plugin.php',
             'events' => ['OnManagerPageBeforeRender'],
         ],
		 'i18nLangRoute' => [
             'file' => 'i18nLangRoute.plugin.php',
             'events' => ['OnHandleRequest'],
         ],
    ],
];

$snippetsDir = $sources['elements'] . 'snippets' . DIRECTORY_SEPARATOR;
$pluginsDir  = $sources['elements'] . 'plugins' . DIRECTORY_SEPARATOR;

// Create category container
/** @var modCategory $category */
$category = $modx->newObject(modCategory::class);
$category->fromArray([
    'category' => 'i18n',
], '', true, true);

// ---- Snippets (attach to category)
$snippetObjects = [];
foreach ((array)($elements['snippets'] ?? []) as $snippetName => $file) {
    $filePath = $snippetsDir . (string)$file;
    $content = $readFile($filePath);
	$content = $normalizePhpElement($content);

    if (trim($content) === '') {
        $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] Build: snippet skipped (empty): ' . (string)$snippetName);
        continue;
    }

    /** @var modSnippet $snippet */
    $snippet = $modx->newObject(modSnippet::class);
    $snippet->fromArray([
        'name'        => (string)$snippetName,
        'description' => 'i18n snippet: ' . (string)$snippetName,
        'snippet'     => $content,
        'static'      => false,
    ], '', true, true);

    $snippetObjects[] = $snippet;
    $modx->log(modX::LOG_LEVEL_INFO, '[i18n] Build: snippet prepared: ' . (string)$snippetName);
}

if (!empty($snippetObjects)) {
    $category->addMany($snippetObjects, 'Snippets');
}

// ---- Plugins + Events (attach to category)
$pluginObjects = [];
foreach ((array)($elements['plugins'] ?? []) as $pluginName => $meta) {
    $file   = (string)($meta['file'] ?? '');
    $events = (array)($meta['events'] ?? []);

    $filePath = $pluginsDir . $file;
    $content = $readFile($filePath);
	$content = $normalizePhpElement($content);

    if (trim($content) === '') {
        $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] Build: plugin skipped (empty): ' . (string)$pluginName);
        continue;
    }

    /** @var modPlugin $plugin */
    $plugin = $modx->newObject(modPlugin::class);
    $plugin->fromArray([
        'name'        => (string)$pluginName,
        'description' => 'i18n plugin: ' . (string)$pluginName,
        'plugincode'  => $content,
        'disabled'    => false,
        'static'      => false,
    ], '', true, true);

    // Create plugin events
    $eventObjects = [];
    foreach ($events as $evt) {
        $evt = trim((string)$evt);
        if ($evt === '') {
            continue;
        }

        /** @var modPluginEvent $pe */
        $pe = $modx->newObject(modPluginEvent::class);
        $pe->fromArray([
            'event'       => $evt,
            'priority'    => 0,
            'propertyset' => 0,
        ], '', true, true);

        $eventObjects[] = $pe;
    }

    // IMPORTANT: specify the relation alias explicitly
    if (!empty($eventObjects)) {
        $plugin->addMany($eventObjects, 'PluginEvents');
    }

    $pluginObjects[] = $plugin;

    $modx->log(modX::LOG_LEVEL_INFO, '[i18n] Build: plugin prepared: ' . (string)$pluginName . ' (events=' . count($eventObjects) . ')');
}

if (!empty($pluginObjects)) {
    $category->addMany($pluginObjects, 'Plugins');
}

// Create single category vehicle with nested relations
$categoryVehicle = $builder->createVehicle($category, [
    xPDOTransport::UNIQUE_KEY    => 'category',
    xPDOTransport::PRESERVE_KEYS => true,
    xPDOTransport::UPDATE_OBJECT => true,
    xPDOTransport::RELATED_OBJECTS => true,
    xPDOTransport::RELATED_OBJECT_ATTRIBUTES => [
        'Snippets' => [
            xPDOTransport::UNIQUE_KEY    => 'name',
            xPDOTransport::PRESERVE_KEYS => false,
            xPDOTransport::UPDATE_OBJECT => true,
        ],
        'Plugins' => [
            xPDOTransport::UNIQUE_KEY    => 'name',
            xPDOTransport::PRESERVE_KEYS => false,
            xPDOTransport::UPDATE_OBJECT => true,
            xPDOTransport::RELATED_OBJECTS => true,
            xPDOTransport::RELATED_OBJECT_ATTRIBUTES => [
                'PluginEvents' => [
                    // MODX 3/xPDO: composite unique key must be an array
                    xPDOTransport::UNIQUE_KEY    => ['pluginid', 'event'],
                    xPDOTransport::PRESERVE_KEYS => false,
                    xPDOTransport::UPDATE_OBJECT => true,
                ],
            ],
        ],
    ],
]);

$builder->putVehicle($categoryVehicle);
$modx->log(modX::LOG_LEVEL_INFO, '[i18n] Build: category vehicle packed: i18n');

// ----------
// Menu (MODX 3): topnav
// ----------
$menu = $modx->newObject(modMenu::class);
$menu->fromArray([
    'text'        => 'i18n',
    'parent'      => 'topnav',
    'description' => 'i18n multilingual system',
    'icon'        => '<i class="icon-language icon"></i>',
    'menuindex'   => 4,
    'namespace'   => 'i18n',
    'action'      => 'index',
], '', true, true);

$menuVehicle = $builder->createVehicle($menu, [
    xPDOTransport::UNIQUE_KEY    => 'text',
    xPDOTransport::PRESERVE_KEYS => true,
    xPDOTransport::UPDATE_OBJECT => true,
]);
$builder->putVehicle($menuVehicle);

$docsDir = __DIR__ . '/docs/';

$builder->setPackageAttributes([
    'license'   => $readDoc($docsDir . 'LICENSE.txt'),
    'readme'    => $readDoc($docsDir . 'README.txt'),
    'changelog' => $readDoc($docsDir . 'CHANGELOG.txt'),
    'requirements' => $readDoc($docsDir . 'REQUIREMENTS.txt'),
]);

$builder->pack();

$modx->log(modX::LOG_LEVEL_INFO, 'Build completed in ' . round(microtime(true) - $mtime, 3) . 's');
exit(0);
