<?php
/**
 * Create DB tables for i18n package via raw SQL (reliable fallback).
 * Works regardless of xPDO model loading.
 */

use MODX\Revolution\modX;
use xPDO\Transport\xPDOTransport;

if (!isset($modx) || !($modx instanceof modX)) {
    return true;
}

$action = (int)($options[xPDOTransport::PACKAGE_ACTION] ?? 0);
if (!in_array($action, [xPDOTransport::ACTION_INSTALL, xPDOTransport::ACTION_UPGRADE], true)) {
    return true;
}

$prefix = (string)$modx->getOption('table_prefix', null, 'modx_');

// Defensive: allow only [A-Za-z0-9_]
$prefix = preg_replace('/[^A-Za-z0-9_]/', '', $prefix);

$tblKeys   = $prefix . 'i18n_keys';
$tblValues = $prefix . 'i18n_values';
$tblUsage  = $prefix . 'i18n_usage';

$modx->log(modX::LOG_LEVEL_INFO, "[i18n] Creating tables via SQL. Prefix={$prefix}");

$sql = [];

// KEYS
$sql[] = "
CREATE TABLE IF NOT EXISTS `{$tblKeys}` (
  `id` int unsigned NOT NULL AUTO_INCREMENT,
  `key` varchar(191) CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci NOT NULL,
  `group` varchar(64) CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci DEFAULT NULL,
  `description` varchar(255) CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci DEFAULT NULL,
  `hash_default` char(32) CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci DEFAULT NULL,
  `usage_count` int unsigned NOT NULL DEFAULT '0',
  `created_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  `last_used_at` datetime DEFAULT NULL,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uniq_key` (`key`),
  KEY `idx_last_used_at` (`last_used_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;
";

// VALUES
$sql[] = "
CREATE TABLE IF NOT EXISTS `{$tblValues}` (
  `id` int unsigned NOT NULL AUTO_INCREMENT,
  `key_id` int unsigned NOT NULL,
  `lang` varchar(16) CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci NOT NULL,
  `value` mediumtext CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci,
  `hash` char(32) CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci DEFAULT NULL,
  `updated_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `uniq_key_lang` (`key_id`,`lang`),
  KEY `idx_lang` (`lang`),
  CONSTRAINT `fk_i18n_values_key` FOREIGN KEY (`key_id`)
    REFERENCES `{$tblKeys}` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;
";

// USAGE
$sql[] = "
CREATE TABLE IF NOT EXISTS `{$tblUsage}` (
  `id` int unsigned NOT NULL AUTO_INCREMENT,
  `key_id` int unsigned NOT NULL,
  `context_type` varchar(32) NOT NULL,
  `context_id` int unsigned DEFAULT NULL,
  `context_name` varchar(191) DEFAULT NULL,
  `field` varchar(64) NOT NULL,
  `preview` varchar(255) DEFAULT NULL,
  `last_scan_at` datetime NOT NULL,
  PRIMARY KEY (`id`),
  KEY `idx_key` (`key_id`),
  KEY `idx_type` (`context_type`),
  CONSTRAINT `fk_i18n_usage_key` FOREIGN KEY (`key_id`)
    REFERENCES `{$tblKeys}` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci;
";

try {
    /** @var PDO $pdo */
    $pdo = $modx->prepare('SELECT 1') ? $modx->getConnection() : null; // not used directly; ensures connection
} catch (Throwable $e) {
    $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] DB connection failed: ' . $e->getMessage());
    return true;
}

foreach ($sql as $q) {
    try {
        $stmt = $modx->prepare($q);
        if (!$stmt) {
            $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] SQL prepare failed.');
            continue;
        }
        $ok = $stmt->execute();
        if (!$ok) {
            $err = $stmt->errorInfo();
            $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] SQL execute failed: ' . json_encode($err));
        }
    } catch (Throwable $e) {
        $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] SQL exception: ' . $e->getMessage());
    }
}

// Final verification
try {
    $check = $modx->query("SHOW TABLES LIKE " . $modx->quote($tblKeys));
    $hasKeys = $check && $check->fetchColumn() ? true : false;

    $check = $modx->query("SHOW TABLES LIKE " . $modx->quote($tblValues));
    $hasValues = $check && $check->fetchColumn() ? true : false;

    $check = $modx->query("SHOW TABLES LIKE " . $modx->quote($tblUsage));
    $hasUsage = $check && $check->fetchColumn() ? true : false;

    $modx->log(modX::LOG_LEVEL_INFO, "[i18n] Tables present: keys=" . ($hasKeys?'yes':'no') . ", values=" . ($hasValues?'yes':'no') . ", usage=" . ($hasUsage?'yes':'no'));
} catch (Throwable $e) {
    $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] Verification failed: ' . $e->getMessage());
}

return true;
