<?php
/**
 * License resolver:
 * - ensure license_state exists and has fingerprint
 * - do not overwrite existing license_state/license_payload (handled by UPDATE_OBJECT=false in build)
 */

use MODX\Revolution\modX;
use MODX\Revolution\modSystemSetting;
use xPDO\Transport\xPDOTransport;

if (!isset($modx) || !($modx instanceof modX)) {
    return true;
}

$action = (int)($options[xPDOTransport::PACKAGE_ACTION] ?? 0);
if (!in_array($action, [xPDOTransport::ACTION_INSTALL, xPDOTransport::ACTION_UPGRADE], true)) {
    return true;
}

function i18n_get_setting(modX $modx, string $key): ?string {
    $s = $modx->getObject(modSystemSetting::class, ['key' => $key]);
    if (!$s) return null;
    $v = $s->get('value');
    return is_string($v) ? $v : (string)$v;
}

function i18n_set_setting(modX $modx, string $key, string $value): bool {
    /** @var modSystemSetting|null $s */
    $s = $modx->getObject(modSystemSetting::class, ['key' => $key]);
    if (!$s) {
        $s = $modx->newObject(modSystemSetting::class);
        $s->fromArray([
            'key' => $key,
            'namespace' => 'i18n',
            'area' => 'license',
            'xtype' => 'textarea',
        ], '', true, true);
    }
    $s->set('value', $value);
    return (bool)$s->save();
}

function i18n_compute_fingerprint(modX $modx): string {
    // Aim: stable per installation, 32-hex like your current fingerprints.
    $uuid = (string)$modx->getOption('uuid', null, '');
    $siteUrl = (string)$modx->getOption('site_url', null, '');
    $httpHost = (string)($_SERVER['HTTP_HOST'] ?? '');
    $docRoot = (string)($_SERVER['DOCUMENT_ROOT'] ?? '');
    $salt = $uuid . '|' . $siteUrl . '|' . $httpHost . '|' . $docRoot . '|' . PHP_VERSION;
    return md5(hash('sha256', $salt, false));
}

$stateKey = 'i18n.license_state';

// Read current state (if any)
$raw = i18n_get_setting($modx, $stateKey);
$state = [];

if (is_string($raw) && $raw !== '') {
    $decoded = json_decode($raw, true);
    if (is_array($decoded)) {
        $state = $decoded;
    }
}

if (empty($state['fingerprint']) || !is_string($state['fingerprint'])) {
    $state['fingerprint'] = i18n_compute_fingerprint($modx);
}

// Normalized defaults (optional, but useful for UI)
$state['last_checkin_at'] = (int)($state['last_checkin_at'] ?? 0);
$state['last_checkin_ok'] = (bool)($state['last_checkin_ok'] ?? false);
$state['last_ok_at']      = (int)($state['last_ok_at'] ?? 0);

i18n_set_setting($modx, $stateKey, json_encode($state, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));

return true;
