<?php
/**
 * Ensure plugin exists and attach events (install/upgrade).
 *
 * If plugin is missing at resolver time, attempts to create it from file
 * inside installed component (after file resolver copied files).
 */

declare(strict_types=1);

use MODX\Revolution\modX;
use MODX\Revolution\modPlugin;
use MODX\Revolution\modPluginEvent;
use MODX\Revolution\modCategory;
use xPDO\Transport\xPDOTransport;

if (!isset($transport) || !is_object($transport)) {
    return false;
}

/** @var modX $modx */
$modx = $transport->xpdo;

$action = $options[xPDOTransport::PACKAGE_ACTION] ?? null;
if (!in_array($action, [xPDOTransport::ACTION_INSTALL, xPDOTransport::ACTION_UPGRADE], true)) {
    return true;
}

// Deterministic config (must match your build config)
$categoryName = 'i18n';

$plugins = [
    'i18nManagerTools' => [
        'file' => 'i18nManagerTools.plugin.php',
        'events' => ['OnManagerPageBeforeRender'],
        'disabled' => false,
        'priority' => 0,
    ],
    'i18nLangRoute' => [
        'file' => 'i18nLangRoute.plugin.php',
        'events' => ['OnHandleRequest'],
        'disabled' => false,
        'priority' => 0,
    ],
];

$createdPlugins = 0;
$createdEvents = 0;

foreach ($plugins as $pluginName => $meta) {
    $pluginName = trim((string)$pluginName);
    if ($pluginName === '') {
        continue;
    }

    /** @var modPlugin|null $plugin */
    $plugin = $modx->getObject(modPlugin::class, ['name' => $pluginName]);

    // If plugin is missing, create it from installed file
    if (!$plugin) {
        $file = (string)($meta['file'] ?? '');
        $file = ltrim($file, '/\\');

        $pluginPath = MODX_CORE_PATH . 'components/i18n/elements/plugins/' . $file;

        if (!is_file($pluginPath)) {
            $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] Plugin events resolver: plugin missing and file not found: ' . $pluginName . ' file=' . $pluginPath);
            // Do not hard-fail whole install, but this plugin won't work
            continue;
        }

        $code = file_get_contents($pluginPath);
        if ($code === false || trim($code) === '') {
            $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] Plugin events resolver: plugin missing and file is empty: ' . $pluginName . ' file=' . $pluginPath);
            continue;
        }

        // Ensure category exists
        /** @var modCategory|null $cat */
        $cat = $modx->getObject(modCategory::class, ['category' => $categoryName]);
        if (!$cat) {
            $cat = $modx->newObject(modCategory::class);
            $cat->set('category', $categoryName);
            $cat->save();
        }

        $plugin = $modx->newObject(modPlugin::class);
        $plugin->fromArray([
            'name' => $pluginName,
            'description' => 'i18n plugin: ' . $pluginName,
            'plugincode' => $code,
            'disabled' => (bool)($meta['disabled'] ?? false),
            'static' => false,
            'category' => (int)$cat->get('id'),
        ], '', true, true);

        if (!$plugin->save()) {
            $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] Plugin events resolver: failed to create plugin: ' . $pluginName);
            continue;
        }

        $createdPlugins++;
        $modx->log(modX::LOG_LEVEL_INFO, '[i18n] Plugin events resolver: plugin created: ' . $pluginName);
    }

    $pluginId = (int)$plugin->get('id');
    if ($pluginId <= 0) {
        $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] Plugin events resolver: invalid plugin id: ' . $pluginName);
        continue;
    }

    $events = array_values(array_filter(array_map('trim', (array)($meta['events'] ?? []))));
    $priority = (int)($meta['priority'] ?? 0);

    foreach ($events as $evt) {
        $exists = $modx->getObject(modPluginEvent::class, [
            'pluginid' => $pluginId,
            'event' => $evt,
        ]);

        if ($exists) {
            continue;
        }

        $pe = $modx->newObject(modPluginEvent::class);
        $pe->fromArray([
            'pluginid' => $pluginId,
            'event' => $evt,
            'priority' => $priority,
            'propertyset' => 0,
        ], '', true, true);

        if ($pe->save()) {
            $createdEvents++;
            $modx->log(modX::LOG_LEVEL_INFO, '[i18n] Plugin events resolver: event attached: ' . $pluginName . ' → ' . $evt);
        } else {
            $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] Plugin events resolver: failed to attach event: ' . $pluginName . ' → ' . $evt);
        }
    }
}

$modx->log(modX::LOG_LEVEL_INFO, '[i18n] Plugin events resolver: createdPlugins=' . $createdPlugins . ', createdEvents=' . $createdEvents);
return true;
