<?php
/**
 * Ensure i18n DB tables using xPDO model classes (no raw schema SQL).
 *
 * Supports metadata.mysql.php with structure:
 * - version, namespace, namespacePrefix
 * - class_map => [ baseClass => [ class1, class2, ... ], ... ]
 *
 * Strategy:
 * - include metadata.mysql.php
 * - flatten class_map to class list
 * - load each class explicitly (multiple filename conventions supported)
 * - createObjectContainer() for each class
 *
 * Runs on install and upgrade.
 */

declare(strict_types=1);

use MODX\Revolution\modX;
use xPDO\Transport\xPDOTransport;

if (!isset($transport) || !is_object($transport)) {
    return false;
}

/** @var modX $modx */
$modx = $transport->xpdo;

$action = $options[xPDOTransport::PACKAGE_ACTION] ?? null;
if (!in_array($action, [xPDOTransport::ACTION_INSTALL, xPDOTransport::ACTION_UPGRADE], true)) {
    $modx->log(modX::LOG_LEVEL_INFO, '[i18n] xPDO tables resolver: skip (action=' . (string)$action . ')');
    return true;
}

$modx->log(modX::LOG_LEVEL_INFO, '[i18n] xPDO tables resolver: start (action=' . (string)$action . ')');

// Model base: core/components/i18n/model/
$modelBase = MODX_CORE_PATH . 'components/i18n/model/';
if (!is_dir($modelBase)) {
    $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] xPDO tables resolver: model base not found: ' . $modelBase);
    return false;
}

// Locate metadata file: model/<pkg>/metadata.mysql.php
$metaFile = null;
$pkgDir   = null;

$dirs = glob($modelBase . '*', GLOB_ONLYDIR) ?: [];
foreach ($dirs as $dir) {
    $candidate = rtrim($dir, '/\\') . DIRECTORY_SEPARATOR . 'metadata.mysql.php';
    if (is_file($candidate)) {
        $metaFile = $candidate;
        $pkgDir = rtrim($dir, '/\\') . DIRECTORY_SEPARATOR;
        break;
    }
}

if (!$metaFile || !$pkgDir) {
    $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] xPDO tables resolver: metadata.mysql.php not found under: ' . $modelBase);
    return false;
}

$pkgName = basename(rtrim($pkgDir, '/\\'));
$modx->log(modX::LOG_LEVEL_INFO, '[i18n] xPDO tables resolver: detected package="' . $pkgName . '" meta=' . $metaFile);

// Register package (required for xPDO)
$modx->addPackage($pkgName, $modelBase);

$manager = $modx->getManager();
if (!$manager) {
    $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] xPDO tables resolver: cannot get xPDO manager');
    return false;
}

// Include metadata (defines $xpdo_meta_map)
$xpdo_meta_map = null;

try {
    /** @noinspection PhpIncludeInspection */
    include $metaFile;
} catch (\Throwable $e) {
    $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] xPDO tables resolver: failed to include metadata: ' . $e->getMessage());
    return false;
}

if (!is_array($xpdo_meta_map) || empty($xpdo_meta_map)) {
    $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] xPDO tables resolver: xpdo_meta_map missing/empty after including: ' . $metaFile);
    return false;
}

$classMap = $xpdo_meta_map['class_map'] ?? null;
if (!is_array($classMap) || empty($classMap)) {
    $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] xPDO tables resolver: class_map missing/empty in metadata: ' . $metaFile);
    $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] Keys present: ' . implode(', ', array_keys($xpdo_meta_map)));
    return false;
}

// Flatten class_map: it can be [baseClass => [class1, class2...]]
$classes = [];
foreach ($classMap as $base => $list) {
    if (is_array($list)) {
        foreach ($list as $c) {
            if (is_string($c) && $c !== '') {
                $classes[] = $c;
            }
        }
    } elseif (is_string($list) && $list !== '') {
        $classes[] = $list;
    }
}

$classes = array_values(array_unique($classes));

$modx->log(modX::LOG_LEVEL_INFO, '[i18n] xPDO tables resolver: classes found=' . count($classes));
if (empty($classes)) {
    $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] xPDO tables resolver: no classes discovered from class_map');
    return false;
}

/**
 * Try to load a class by typical xPDO model conventions.
 * Works even if Composer/PSR-4 is not active during installer stage.
 */
$tryLoad = static function (modX $modx, string $class, string $pkgDir): bool {
    if (class_exists($class)) {
        return true;
    }

    // 1) Try MODX/xPDO loader
    if ($modx->loadClass($class, $pkgDir, true, true)) {
        if (class_exists($class)) {
            return true;
        }
    }

    // 2) Try common filename conventions based on the short class name
    $short = $class;
    if (strpos($short, '\\') !== false) {
        $parts = explode('\\', $short);
        $short = end($parts) ?: $class;
    }

    $candidates = [
        $pkgDir . $short . '.php',
        $pkgDir . $short . '.class.php',
        $pkgDir . strtolower($short) . '.php',
        $pkgDir . strtolower($short) . '.class.php',
    ];

    foreach ($candidates as $file) {
        if (is_file($file)) {
            require_once $file;
            if (class_exists($class)) {
                return true;
            }
        }
    }

    return class_exists($class);
};

$ensured = 0;

foreach ($classes as $class) {
    $class = (string)$class;

    if (!$tryLoad($modx, $class, $pkgDir)) {
        $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] xPDO tables resolver: class not found and cannot be loaded: ' . $class);
        $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] Checked dir: ' . $pkgDir);
        return false;
    }

    try {
        $ok = $manager->createObjectContainer($class);
        if (!$ok) {
            $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] xPDO tables resolver: createObjectContainer failed: ' . $class);
            return false;
        }

        $ensured++;
        $modx->log(modX::LOG_LEVEL_INFO, '[i18n] xPDO tables resolver: ensured container for ' . $class);
    } catch (\Throwable $e) {
        $modx->log(modX::LOG_LEVEL_ERROR, '[i18n] xPDO tables resolver: exception for ' . $class . ': ' . $e->getMessage());
        return false;
    }
}

$modx->log(modX::LOG_LEVEL_INFO, '[i18n] xPDO tables resolver: done, ensured=' . $ensured);
return true;
