<?php

use I18n\Service\Licensing;

class I18nIndexManagerController extends modExtraManagerController
{
    public function getLanguageTopics()
    {
        return ['i18n:default'];
    }

    public function getPageTitle()
    {
        return $this->modx->lexicon('i18n_management');
    }

    public function loadCustomCssJs()
    {
        $assetsUrl = $this->modx->getOption(
            'i18n.assets_url',
            null,
            $this->modx->getOption('assets_url') . 'components/i18n/'
        );

        // CSS for manager UI
        $this->addCss($assetsUrl . 'css/mgr/styles.css');

        // Core JS for manager UI
        $this->addJavascript($assetsUrl . 'js/mgr/i18n.js');
        $this->addJavascript($assetsUrl . 'js/mgr/keys.grid.js');
        $this->addJavascript($assetsUrl . 'js/mgr/key.window.js');
        $this->addJavascript($assetsUrl . 'js/mgr/usage.window.js');
        $this->addJavascript($assetsUrl . 'js/mgr/export.window.js');
        $this->addJavascript($assetsUrl . 'js/mgr/key.update.window.js');
        $this->addJavascript($assetsUrl . 'js/mgr/import.panel.js');
        $this->addJavascript($assetsUrl . 'js/mgr/settings.panel.js');
        $this->addJavascript($assetsUrl . 'js/mgr/home.panel.js');
        $this->addJavascript($assetsUrl . 'js/mgr/home.page.js');

        // Read languages from system settings
        $defaultLang = trim((string)$this->modx->getOption('i18n.default_lang', null, 'en'));
        $allowed     = trim((string)$this->modx->getOption('i18n.allowed_langs', null, ''));

        $allowedLangs = [];
        if ($allowed !== '') {
            $allowedLangs = array_filter(array_map('trim', explode(',', $allowed)));
        }

        // Guarantee at least one language
        if (empty($allowedLangs)) {
            $fallback = $defaultLang !== '' ? $defaultLang : 'en';
            $allowedLangs = [$fallback];
        }

        // Ensure defaultLang is inside list (if set)
        if ($defaultLang !== '' && !in_array($defaultLang, $allowedLangs, true)) {
            $allowedLangs[] = $defaultLang;
        }

        $allowedLangs = array_values(array_unique($allowedLangs));

        // Licensing status for UI gating (server remains the source of truth)
        $license = null;
        try {
            $lic = new Licensing($this->modx);
            $license = $lic->getStatus();
        } catch (Throwable $e) {
            // If licensing fails unexpectedly, keep UI usable but do NOT grant write permissions.
			$license = [
			  'mode' => 'readonly',
			  'write_allowed' => false,
			  'reason' => 'licensing-error',
			  'fingerprint' => '',
			  'trial_left_seconds' => 0,
			  'grace_left_seconds' => 0,
			];
        }

        $config = [
            'connectorUrl' => $assetsUrl . 'connector.php',
            'namespace'    => 'i18n',
            'defaultLang'  => $defaultLang !== '' ? $defaultLang : 'en',
            // Raw list from system settings
            'allowedLangs' => $allowedLangs,
            // Convenience alias used by UI code
            'languages'    => $allowedLangs,

            // Licensing payload for client-side gating
            'license' => $license,
        ];

        $this->modx->lexicon->load('i18n:default');

        $this->addHtml(
            '<script type="text/javascript">
                window.i18n = window.i18n || {};
                i18n.config = i18n.config || {};

                // Do not override existing connectorUrl/namespace if they were provided by i18n.js
                Ext.applyIf(i18n.config, {
                    connectorUrl: ' . json_encode($config['connectorUrl'], JSON_UNESCAPED_SLASHES) . ',
                    namespace: ' . json_encode($config['namespace'], JSON_UNESCAPED_SLASHES) . '
                });

                // Always update language-related configuration + licensing status
                Ext.apply(i18n.config, ' . json_encode([
                    'defaultLang'  => $config['defaultLang'],
                    'allowedLangs' => $config['allowedLangs'],
                    'languages'    => $config['languages'],
                    'license'      => $config['license'],
                ], JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . ');
            </script>'
        );

        $this->addHtml('
        <script type="text/javascript">
        Ext.onReady(function () {
            MODx.load({
                xtype: "i18n-page-home"
            });
        });
        </script>');

        $this->addHtml('<script>
            window.I18N_LEXICON = ' . json_encode(
                $this->modx->lexicon->fetch('i18n.'),
                JSON_UNESCAPED_UNICODE
            ) . ';
        </script>');
    }

    public function getTemplateFile()
    {
        return '';
    }
}
