/**
 * Export window for i18n keys.
 *
 * @class i18n.window.Export
 * @extends MODx.Window
 * @xtype i18n-window-export
 */

i18n.window = i18n.window || {};

i18n.window.Export = function (config) {
    config = config || {};

    this.grid        = config.grid || null;
    this.availableGroups = config.availableGroups || [];
    this.hasSelection    = !!config.hasSelection;

    Ext.applyIf(config, {
        title: i18n.lex('i18n_export', 'Export'),
        id: Ext.id(),
        width: 420,
        autoHeight: true,
        modal: true,
        closeAction: 'close',
        resizable: false,
        maximizable: false,
        cls: 'modx-window',

        items: [{
            xtype: 'form',
            layout: 'form',
            autoHeight: true,
            border: false,
            bodyStyle: 'padding: 10px;',
            labelAlign: 'top',
            items: this.getFields(config)
        }],

        buttons: [{
            text: i18n.lex('i18n_export', 'Export'),
            cls: 'primary-button',
            handler: this.doExport,
            scope: this
        }, {
            text: _('cancel'),
            handler: function () { this.close(); },
            scope: this
        }]
    });

    i18n.window.Export.superclass.constructor.call(this, config);
};

Ext.extend(i18n.window.Export, MODx.Window, {

    /**
     * Build form fields for export options.
     */
    getFields: function () {
        var fields = [];

        // Scope options: Selected / All / Group: xxx
        var scopeStoreData = [];
        var defaultScope   = 'all';

        if (this.hasSelection) {
            scopeStoreData.push([
                'selected',
                i18n.lex('i18n_export_scope_selected', 'Selected rows')
            ]);
            defaultScope = 'selected';
        }

        scopeStoreData.push([
            'all',
            i18n.lex('i18n_export_scope_all', 'All keys')
        ]);

        Ext.each(this.availableGroups, function (g) {
            if (!g) return;
            scopeStoreData.push([
                'group:' + g,
                i18n.lex('i18n_export_scope_group', 'Group') + ': ' + g
            ]);
        });

        fields.push({
            xtype: 'combo',
            fieldLabel: i18n.lex('i18n_export_scope', 'Scope'),
            name: 'scope',
            hiddenName: 'scope',
            anchor: '100%',
            mode: 'local',
            triggerAction: 'all',
            editable: false,
            allowBlank: false,
            value: defaultScope,
            store: new Ext.data.ArrayStore({
                fields: ['value', 'label'],
                data: scopeStoreData
            }),
            valueField: 'value',
            displayField: 'label'
        });

        // Format: JSON / CSV
        fields.push({
            xtype: 'combo',
            fieldLabel: i18n.lex('i18n_export_format', 'Format'),
            name: 'format',
            hiddenName: 'format',
            anchor: '100%',
            mode: 'local',
            triggerAction: 'all',
            editable: false,
            allowBlank: false,
            value: 'json',
            store: new Ext.data.ArrayStore({
                fields: ['value', 'label'],
                data: [
                    ['json', 'JSON'],
                    ['csv', 'CSV']
                ]
            }),
            valueField: 'value',
            displayField: 'label'
        });

        // Language checkboxes: allowedLangs + defaultLang (unique, default first)
        var langOptions = this.getLangOptions();
        if (langOptions.length) {
            var langCheckboxes = [];

            Ext.each(langOptions, function (code) {
                langCheckboxes.push({
                    xtype: 'xcheckbox',
                    boxLabel: code,
                    name: 'lang_' + code,
                    inputValue: code,
                    checked: true
                });
            });

            fields.push({
                xtype: 'fieldset',
                title: i18n.lex('i18n_export_languages', 'Languages'),
                autoHeight: true,
                collapsible: false,
                items: langCheckboxes
            });
        }

        return fields;
    },

    /**
     * Build language list based on config.allowedLangs + defaultLang.
     *
     * Default language is always first, duplicates removed.
     */
    getLangOptions: function () {
        var cfg = i18n.config || {};
        var raw = [];

        if (Ext.isArray(cfg.allowedLangs)) {
            raw = cfg.allowedLangs.slice();
        } else if (typeof cfg.allowedLangs === 'string') {
            raw = cfg.allowedLangs.split(',');
        }

        var defaultLang = cfg.defaultLang || 'en';

        // Normalize and remove duplicates
        var seen = {};
        var langs = [];

        Ext.each(raw, function (l) {
            var code = String(l || '').trim();
            if (!code || seen[code]) return;
            seen[code] = true;
            langs.push(code);
        });

        if (defaultLang) {
            defaultLang = String(defaultLang).trim();
            if (defaultLang && !seen[defaultLang]) {
                langs.unshift(defaultLang);
            } else if (defaultLang && seen[defaultLang]) {
                // Move default to the first position if it is already in the list
                var idx = langs.indexOf(defaultLang);
                if (idx > 0) {
                    langs.splice(idx, 1);
                    langs.unshift(defaultLang);
                }
            }
        }

        return langs;
    },

    /**
     * Perform export using grid helper.
     */
    doExport: function () {
        if (!this.grid) {
            this.close();
            return;
        }

        var formPanel = this.items.get(0);
        if (!formPanel || !formPanel.getForm) {
            this.close();
            return;
        }

        var form  = formPanel.getForm();
        var vals  = form.getValues();
        var scope = vals.scope || 'all';
        var fmt   = vals.format || 'json';

        var exportCfg = {
            format: fmt
        };

        // Scope: selected / all / group:xxx
        if (scope === 'selected') {
            var sm  = this.grid.getSelectionModel();
            var sel = sm && sm.getSelections ? sm.getSelections() : [];
            var ids = [];

            Ext.each(sel, function (rec) {
                var id = rec.get('id');
                if (id) ids.push(id);
            });

            if (!ids.length) {
                MODx.msg.alert(
                    i18n.lex('i18n_export', 'Export'),
                    i18n.lex('i18n_export_selected_empty', 'No selected rows to export.')
                );
                return;
            }

            exportCfg.scope = 'selected';
            exportCfg.ids   = ids.join(',');
        } else if (scope.indexOf('group:') === 0) {
            var groupName = scope.substring('group:'.length);
            exportCfg.scope = 'group';
            exportCfg.group = groupName;
        } else {
            exportCfg.scope = 'all';
        }

        // Languages: collect checked lang_* fields
        var langOptions = this.getLangOptions();
        var selectedLangs = [];

        Ext.each(langOptions, function (code) {
            var f = form.findField('lang_' + code);
            if (f && f.getValue()) {
                selectedLangs.push(code);
            }
        });

        if (selectedLangs.length) {
            exportCfg.langs = selectedLangs.join(',');
        }

        this.grid.exportKeys(exportCfg);
        this.close();
    }
});

Ext.reg('i18n-window-export', i18n.window.Export);