/**
 * Main page for i18n manager.
 *
 * @class i18n.page.Home
 * @extends MODx.Component
 * @xtype i18n-page-home
 */

// We assume i18n singleton is already created in i18n.js
i18n.page = i18n.page || {};

/**
 * @param {Object} config
 * @constructor
 */
i18n.page.Home = function (config) {
    config = config || {};

    Ext.applyIf(config, {
        components: [{
            xtype: 'i18n-panel-home'
        }]
    });

    i18n.page.Home.superclass.constructor.call(this, config);
};

/**
 * Import workflow (preview/import).
 *
 * Server (preview) must return:
 *  {success:true, object:{groups:[], preview_html:"..."}}
 *
 * Server (import) should return:
 *  {success:true, object:{keys_created:0, keys_updated:0, values_written:0}}
 */
i18n.runImport = function (mode) {
    var formCmp = Ext.getCmp('i18n-import-form');
    if (!formCmp || !formCmp.getForm) {
        MODx.msg.alert(_('error'), i18n.lex('i18n_import_form_not_available', 'Import form is not available.'));
        return;
    }

    var form = formCmp.getForm();

    if (!form.isValid()) {
        MODx.msg.alert(_('error'), i18n.lex('i18n_import_select_file', 'Please select a file first.'));
        return;
    }

    // Strategy
    var groupStrategyField = form.findField('group_strategy');
    var groupStrategy = groupStrategyField ? groupStrategyField.getValue() : 'asis';

    // Collect languages: lang_xx checkboxes
    var rawValues = form.getValues();
    var selectedLangs = [];

    for (var k in rawValues) {
        if (!rawValues.hasOwnProperty(k)) continue;
        if (k.indexOf('lang_') !== 0) continue;

        var code = k.substring(5);
        var v = rawValues[k];
        if (v && (v === true || v === 'true' || v === 'on' || v === 1 || v === '1')) {
            selectedLangs.push(code);
        }
    }

    var params = {
        action: 'mgr/import',
        namespace: i18n.config.namespace || 'i18n',
        mode: mode,
        format: (form.findField('format') ? form.findField('format').getValue() : 'json'),
        group_strategy: groupStrategy,
        single_group: (form.findField('single_group') ? (form.findField('single_group').getValue() || '') : ''),
        custom_group: (form.findField('custom_group') ? (form.findField('custom_group').getValue() || '') : ''),
        overwrite: (form.findField('overwrite') ? form.findField('overwrite').getValue() : 'all'),

        // IMPORTANT for iframe upload responses
        isUpload: 1
    };

    if (selectedLangs.length) {
        params.langs = selectedLangs.join(',');
    }

    form.submit({
        url: i18n.config.connectorUrl,
        params: params,
        waitMsg: mode === 'import'
            ? i18n.lex('i18n_import_wait', 'Importing translations…')
            : i18n.lex('i18n_import_preview_wait', 'Preparing preview…'),

        success: function (fp, r) {
            var panel = Ext.getCmp('i18n-import-preview-panel');
            if (!panel) return;

            // MODX responses can vary. Normalize safely.
            var obj = null;

            if (r && r.result) {
                if (r.result.object) obj = r.result.object;
                else if (r.result.data) obj = r.result.data;
            }
            if (!obj && r && r.object) obj = r.object;
            if (!obj) obj = {};

            if (mode === 'preview') {
                var groups = Ext.isArray(obj.groups) ? obj.groups : [];
                var previewHtml = obj.preview_html || '';

                // Fill Target group dropdown
                var combo = Ext.getCmp('i18n-import-single-group');
                if (combo && combo.getStore) {
                    var store = combo.getStore();
                    store.removeAll();

                    if (groups.length) {
                        store.loadData(groups.map(function (g) { return [g]; }));

                        // Auto-select first group if empty
                        if (!combo.getValue()) {
                            combo.setValue(groups[0]);
                        }
                    }
                }

                panel.update(previewHtml || '<p>' + i18n.lex('i18n_import_no_preview', 'No preview data.') + '</p>');
                panel.show();
                return;
            }

            // Import completed
            var msg =
                '<p style="margin:0 0 10px 0;">' +
                i18n.lex('i18n_import_done', 'Import completed.') +
                '</p>' +
                '<ul style="margin:0; padding-left:18px;">' +
                '<li>Keys created: ' + (obj.keys_created || 0) + '</li>' +
                '<li>Keys updated: ' + (obj.keys_updated || 0) + '</li>' +
                '<li>Values written: ' + (obj.values_written || 0) + '</li>' +
                '</ul>';

            panel.update(msg);
            panel.show();

            // Refresh grid
            var grid = Ext.getCmp('i18n-grid-keys');
            if (grid && grid.refresh) {
                grid.refresh();
            }
        },

        failure: function (fp, r) {
            MODx.msg.alert(
                _('error'),
                (r && r.result && r.result.message)
                    ? r.result.message
                    : i18n.lex('i18n_import_error', 'Import failed.')
            );
        },

        scope: this
    });
};

/**
 * Enable/disable group fields depending on group strategy.
 *
 * @param {String} strategy "asis" | "single" | "custom"
 */
i18n.importApplyGroupStrategy = function (strategy) {
    strategy = String(strategy || 'asis');

    var singleField = Ext.getCmp('i18n-import-single-group');
    var customField = Ext.getCmp('i18n-import-custom-group');

    if (singleField) {
        singleField.setDisabled(strategy !== 'single');
        if (strategy !== 'single') {
            // Do not clear value automatically; keep it for user convenience
            // singleField.setValue('');
        }
    }

    if (customField) {
        customField.setDisabled(strategy !== 'custom');
        if (strategy !== 'custom') {
            // customField.setValue('');
        }
    }
};

Ext.extend(i18n.page.Home, MODx.Component);
Ext.reg('i18n-page-home', i18n.page.Home);
