/**
 * Main panel for i18n manager.
 *
 * @class i18n.panel.Home
 * @extends MODx.Panel
 * @xtype i18n-panel-home
 */
i18n = window.i18n || {};
i18n.panel = i18n.panel || {};

i18n.panel.Home = function (config) {
    config = config || {};

    Ext.applyIf(config, {
        id: 'i18n-panel-home',
        cls: 'container',
        layout: 'anchor',
        items: [{
            xtype: 'modx-header',
            html: i18n.lex('i18n_management', 'i18n Manager')
        }, {
            // License banner placeholder (filled after status call)
            xtype: 'panel',
            id: 'i18n-license-banner',
            hidden: true,
            border: true,
            cls: 'main-wrapper',
            bodyStyle: 'padding: 10px; margin: 10px 0; background: #fff7e6; border: 1px solid #f2c97d;',
            html: ''
        }, {
            xtype: 'modx-tabs',
            id: 'i18n-main-tabs',
            anchor: '100%',
            border: true,
            defaults: {
                border: false,
                autoHeight: true,
                bodyStyle: 'padding: 15px'
            },
            items: [
                // --- TAB: KEYS ---
                {
                    title: i18n.lex('i18n_tab_keys', 'Keys'),
                    layout: 'form',
                    items: [{
                        xtype: 'modx-description',
                        html: '<p>' + i18n.lex(
                            'i18n_management_desc',
                            'Manage translation keys used across the system.'
                        ) + '</p>'
                    }, {
                        xtype: 'i18n-grid-keys',
                        itemId: 'grid-i18n-keys',
                        cls: 'main-wrapper',
                        anchor: '100%'
                    }]
                },

                // --- TAB: IMPORT ---
                {
                    title: i18n.lex('i18n_tab_import', 'Import'),
                    layout: 'form',
                    autoScroll: true,
                    items: [{
                        xtype: 'modx-description',
                        html:
                            '<p>' +
                            i18n.lex(
                                'i18n_import_desc',
                                'Import translation keys and values from JSON or CSV. ' +
                                'Preview reads the file and shows summary + groups found.'
                            ) +
                            '</p>'
                    }, {
                        // IMPORTANT: keep classic Ext form panel here to preserve your UI look
                        xtype: 'form',
                        fileUpload: true,
                        id: 'i18n-import-form',
                        border: false,
                        labelAlign: 'top',
                        anchor: '100%',
                        bodyStyle: 'padding: 10px; background: #fff;',
                        items: [
                            {
                                xtype: 'fileuploadfield',
                                fieldLabel: i18n.lex('i18n_import_file', 'File'),
                                name: 'file',
                                id: 'i18n-import-file',
                                anchor: '100%',
                                allowBlank: false
                            },
                            {
                                xtype: 'combo',
                                name: 'format',
                                hiddenName: 'format',
                                fieldLabel: i18n.lex('i18n_import_format', 'Format'),
                                anchor: '100%',
                                store: [['json', 'JSON'], ['csv', 'CSV']],
                                mode: 'local',
                                triggerAction: 'all',
                                editable: false,
                                value: 'json'
                            },

                            // Group strategy
                            {
                                xtype: 'combo',
                                name: 'group_strategy',
                                hiddenName: 'group_strategy',
                                id: 'i18n-import-group-strategy',
                                fieldLabel: i18n.lex('i18n_import_group_strategy', 'Group Strategy'),
                                anchor: '100%',
                                store: [
                                    ['asis', i18n.lex('i18n_import_group_asis', 'Keep as is')],
                                    ['single', i18n.lex('i18n_import_group_single', 'Use single group')],
                                    ['custom', i18n.lex('i18n_import_group_custom', 'Custom group name')]
                                ],
                                mode: 'local',
                                triggerAction: 'all',
                                editable: false,
                                value: 'asis',
                                listeners: {
                                    select: function (cb) {
                                        i18n.importApplyGroupStrategy(cb.getValue());
                                    },
                                    afterrender: function (cb) {
                                        i18n.importApplyGroupStrategy(cb.getValue());
                                    }
                                }
                            },

                            // Target group (filled after Preview from file groups)
                            {
                                xtype: 'combo',
                                fieldLabel: i18n.lex('i18n_import_single_group', 'Target group'),
                                id: 'i18n-import-single-group',
                                name: 'single_group',
                                hiddenName: 'single_group',
                                disabled: true,
                                anchor: '100%',
                                mode: 'local',
                                triggerAction: 'all',
                                editable: true,
                                store: new Ext.data.ArrayStore({
                                    fields: ['value'],
                                    data: []
                                }),
                                valueField: 'value',
                                displayField: 'value'
                            },

                            // Custom group
                            {
                                xtype: 'textfield',
                                fieldLabel: i18n.lex('i18n_import_custom_group', 'Custom group'),
                                id: 'i18n-import-custom-group',
                                name: 'custom_group',
                                disabled: true,
                                anchor: '100%'
                            },

                            // Languages (default first)
                            {
                                xtype: 'fieldset',
                                title: i18n.lex('i18n_import_langs', 'Languages to import'),
                                autoHeight: true,
                                items: (function () {
                                    var cfg = i18n.config || {};
                                    var allowed = cfg.allowedLangs || [];
                                    var def = cfg.defaultLang || null;

                                    if (!Ext.isArray(allowed)) allowed = String(allowed).split(',');

                                    var seen = {};
                                    var langs = [];

                                    if (def) {
                                        def = String(def).trim();
                                        if (def && !seen[def]) { seen[def] = true; langs.push(def); }
                                    }

                                    Ext.each(allowed, function (l) {
                                        var code = String(l || '').trim();
                                        if (!code || seen[code]) return;
                                        seen[code] = true;
                                        langs.push(code);
                                    });

                                    return langs.map(function (code) {
                                        return {
                                            xtype: 'xcheckbox',
                                            boxLabel: code + (def && code === def ? ' (default)' : ''),
                                            name: 'lang_' + code,
                                            checked: true
                                        };
                                    });
                                })()
                            },

                            // Overwrite
                            {
                                xtype: 'combo',
                                name: 'overwrite',
                                hiddenName: 'overwrite',
                                fieldLabel: i18n.lex('i18n_import_overwrite', 'Overwrite rules'),
                                anchor: '100%',
                                store: [
                                    ['all', i18n.lex('i18n_import_overwrite_all', 'Overwrite all')],
                                    ['empty', i18n.lex('i18n_import_overwrite_empty', 'Only empty fields')],
                                    ['skip', i18n.lex('i18n_import_overwrite_skip', 'Skip existing')]
                                ],
                                mode: 'local',
                                triggerAction: 'all',
                                editable: false,
                                value: 'all'
                            }
                        ],

                        buttons: [{
                            text: i18n.lex('i18n_import_preview', 'Preview'),
                            handler: function () {
                                var fileField = Ext.getCmp('i18n-import-file');
                                if (!fileField || !fileField.fileInput || !fileField.fileInput.dom || !fileField.fileInput.dom.files || !fileField.fileInput.dom.files[0]) {
                                    MODx.msg.alert(_('error'), i18n.lex('i18n_import_select_file', 'Please select a file first.'));
                                    return;
                                }

                                var formCmp = Ext.getCmp('i18n-import-form');
                                if (!formCmp || !formCmp.getForm) {
                                    MODx.msg.alert(_('error'), i18n.lex('i18n_import_form_not_found', 'Import form not found.'));
                                    return;
                                }

                                var form = formCmp.getForm();
                                var formatField = form.findField('format');
                                var format = formatField ? formatField.getValue() : 'json';

                                // Collect selected languages
                                var raw = form.getValues();
                                var langs = [];
                                for (var k in raw) {
                                    if (!raw.hasOwnProperty(k)) continue;
                                    if (k.indexOf('lang_') !== 0) continue;

                                    var v = raw[k];
                                    if (v === true || v === 'true' || v === 'on' || v === 1 || v === '1') {
                                        langs.push(k.substring(5));
                                    }
                                }

                                var reader = new FileReader();
                                reader.onload = function () {
                                    MODx.Ajax.request({
                                        url: i18n.config.connectorUrl,
                                        params: {
                                            action: 'mgr/importpreview',
                                            namespace: i18n.config.namespace || 'i18n',
                                            format: format,
                                            content: reader.result,
                                            langs: langs.join(',')
                                        },
                                        listeners: {
                                            success: {
                                                fn: function (r) {
                                                    var obj = r.object || {};
                                                    var groups = Ext.isArray(obj.groups) ? obj.groups : [];
                                                    var previewHtml = obj.preview_html || '';

                                                    // Fill Target group with groups found in file
                                                    var combo = Ext.getCmp('i18n-import-single-group');
                                                    if (combo && combo.getStore) {
                                                        var store = combo.getStore();
                                                        store.removeAll();
                                                        if (groups.length) {
                                                            store.loadData(groups.map(function (g) { return [g]; }));
                                                            if (!combo.getValue()) combo.setValue(groups[0]);
                                                        }
                                                    }

                                                    var panel = Ext.getCmp('i18n-import-preview-panel');
                                                    if (panel) {
                                                        panel.update(previewHtml || '<p>No preview data.</p>');
                                                        panel.show();
                                                    }
                                                },
                                                scope: this
                                            },
                                            failure: {
                                                fn: function (r) {
                                                    MODx.msg.alert(_('error'), r.message || i18n.lex('i18n_preview_failed', 'Preview failed.'));
                                                },
                                                scope: this
                                            }
                                        }
                                    });
                                };

                                reader.onerror = function () {
                                    MODx.msg.alert(_('error'), i18n.lex('i18n_browser_read_failed', 'Failed to read file in browser.'));
                                };

                                reader.readAsText(fileField.fileInput.dom.files[0]);
                            }
                        }, {
                            text: i18n.lex('i18n_import_start', 'Import'),
                            cls: 'primary-button',
                            handler: function () {
                                i18n.importSubmit('import');
                            }
                        }]
                    }, {
                        xtype: 'panel',
                        id: 'i18n-import-preview-panel',
                        title: i18n.lex('i18n_import_preview_title', 'Preview'),
                        hidden: true,
                        autoHeight: true,
                        bodyStyle: 'padding: 10px; background: #fafafa; border: 1px solid #ddd;',
                        html: ''
                    }]
                },

                // --- TAB: SETTINGS ---
                {
                    title: i18n.lex('i18n_tab_settings', 'Settings'),
                    layout: 'form',
                    items: [{
                        xtype: 'modx-description',
                        html: '<p>' + i18n.lex(
                            'i18n_settings_desc',
                            'Configure default language, allowed languages and tracking options.'
                        ) + '</p>'
                    }, {
                        xtype: 'i18n-panel-settings',
                        itemId: 'i18n-settings-panel',
                        anchor: '100%'
                    }]
                }
            ]
        }]
    });

    /**
     * Enable/disable group fields depending on group strategy.
     */
    i18n.importApplyGroupStrategy = function (strategy) {
        strategy = String(strategy || 'asis');

        var singleField = Ext.getCmp('i18n-import-single-group');
        var customField = Ext.getCmp('i18n-import-custom-group');

        if (singleField) singleField.setDisabled(strategy !== 'single');
        if (customField) customField.setDisabled(strategy !== 'custom');
    };

    /**
     * Submit import form to connector (preview/import).
     */
    i18n.importSubmit = function (mode) {
        mode = String(mode || 'preview');

        var formCmp = Ext.getCmp('i18n-import-form');
        if (!formCmp || !formCmp.getForm) {
            MODx.msg.alert(_('error'), i18n.lex('i18n_import_form_not_found_id', 'Import form not found.'));
            return;
        }

        var form = formCmp.getForm();
        if (!form.isValid()) {
            MODx.msg.alert(_('error'), 'Please select a file first.');
            return;
        }

        // Collect selected languages
        var raw = form.getValues();
        var langs = [];
        for (var k in raw) {
            if (!raw.hasOwnProperty(k)) continue;
            if (k.indexOf('lang_') !== 0) continue;

            var v = raw[k];
            if (v === true || v === 'true' || v === 'on' || v === 1 || v === '1') {
                langs.push(k.substring(5));
            }
        }

        form.submit({
            url: i18n.config.connectorUrl,
            params: {
                action: 'mgr/import',
                namespace: i18n.config.namespace || 'i18n',

                mode: mode,
                format: form.findField('format') ? form.findField('format').getValue() : 'json',
                group_strategy: form.findField('group_strategy') ? form.findField('group_strategy').getValue() : 'asis',
                single_group: form.findField('single_group') ? (form.findField('single_group').getValue() || '') : '',
                custom_group: form.findField('custom_group') ? (form.findField('custom_group').getValue() || '') : '',
                overwrite: form.findField('overwrite') ? form.findField('overwrite').getValue() : 'all',
                langs: langs.join(','),

                // REQUIRED for iframe upload responses
                isUpload: 1
            },
            waitMsg: (mode === 'preview') ? 'Preparing preview…' : 'Importing…',
            success: function (fp, r) {
                var obj = (r && r.result && r.result.object) ? r.result.object : {};
                var panel = Ext.getCmp('i18n-import-preview-panel');

                if (mode === 'preview') {
                    var groups = Ext.isArray(obj.groups) ? obj.groups : [];
                    var html = obj.preview_html || '';

                    // Fill Target group store with groups from file
                    var combo = Ext.getCmp('i18n-import-single-group');
                    if (combo && combo.getStore) {
                        var store = combo.getStore();
                        store.removeAll();
                        if (groups.length) {
                            store.loadData(groups.map(function (g) { return [g]; }));
                            if (!combo.getValue()) combo.setValue(groups[0]);
                        }
                    }

                    if (panel) {
                        panel.update(html || '<p>' + i18n.lex('i18n_import_no_preview', 'No preview data.') + '</p>');
                        panel.show();
                    }

                    return;
                }

                // Import result
                if (panel) {
                    panel.update('<p>' + i18n.lex('i18n_import_done', 'Import completed.') + '</p>');
                    panel.show();
                }

                var grid = Ext.getCmp('i18n-grid-keys');
                if (grid && grid.refresh) grid.refresh();
            },
            failure: function (fp, r) {
                MODx.msg.alert(_('error'), (r && r.result && r.result.message) ? r.result.message : i18n.lex('i18n_request_failed', 'Request failed.'));
            }
        });
    };

    /**
     * Request license status and update UI (banner + capabilities).
     */
    i18n.refreshLicenseStatus = function (cb) {
        MODx.Ajax.request({
            url: i18n.config.connectorUrl,
            params: {
                action: 'mgr/license/status',
                namespace: i18n.config.namespace || 'i18n'
            },
            listeners: {
                success: {
                    fn: function (r) {
                        var st = (r && r.object) ? r.object : {};
                        i18n.applyLicenseStatus(st);
                        if (typeof cb === 'function') cb(true, st);
                    },
                    scope: this
                },
                failure: {
                    fn: function () {
                        // If status request fails, do not hard-block UI; backend processors still enforce write-gate.
                        if (typeof cb === 'function') cb(false, null);
                    },
                    scope: this
                }
            }
        });
    };

    /**
     * Apply license status to UI.
     *
     * Status payload expected from processor:
     *  - mode: licensed|trial|grace|readonly
     *  - write_allowed: bool
     *  - trial_left_seconds: int|null
     *  - grace_left_seconds: int|null
     */
    i18n.applyLicenseStatus = function (st) {
        st = st || {};
        var mode = String(st.mode || '');
        var writeAllowed = (st.write_allowed === true);

        // Banner
        var banner = Ext.getCmp('i18n-license-banner');
        if (banner) {
            var html = '';
            var show = false;

            function fmtLeft(sec) {
                sec = parseInt(sec || 0, 10);
                if (!sec || sec < 0) sec = 0;
                var days = Math.ceil(sec / 86400);
                if (days <= 0) days = 0;
                return days + ' day' + (days === 1 ? '' : 's');
            }

            if (mode === 'trial') {
                show = true;
                html =
                    '<div style="display:flex; align-items:center; justify-content:space-between; gap:12px;">' +
                        '<div>' +
                            '<strong>Trial mode.</strong> ' +
                            'Full access is enabled. ' +
                            'Time left: <strong>' + fmtLeft(st.ttl) + '</strong>.' +
                        '</div>' +
                        '<div>' +
                            '<button type="button" class="x-btn x-btn-text" onclick="i18n.openLicenseActivateWindow()">Activate license</button>' +
                        '</div>' +
                    '</div>';
            } else if (mode === 'grace') {
                show = true;
                html =
                    '<div style="display:flex; align-items:center; justify-content:space-between; gap:12px;">' +
                        '<div>' +
                            '<strong>Grace period.</strong> ' +
                            'Please activate license. ' +
                            'Time left: <strong>' + fmtLeft(st.grace_ttl) + '</strong>.' +
                        '</div>' +
                        '<div>' +
                            '<button type="button" class="x-btn x-btn-text" onclick="i18n.openLicenseActivateWindow()">Activate license</button>' +
                        '</div>' +
                    '</div>';
            } else if (mode === 'readonly') {
                show = true;
                html =
                    '<div style="display:flex; align-items:center; justify-content:space-between; gap:12px;">' +
                        '<div>' +
                            '<strong>Read-only mode.</strong> ' +
                            'Write operations are blocked until a valid license is activated.' +
                        '</div>' +
                        '<div>' +
                            '<button type="button" class="x-btn x-btn-text" onclick="i18n.openLicenseActivateWindow()">Activate license</button>' +
                        '</div>' +
                    '</div>';
            }

            if (show) {
                banner.update(html);
                banner.show();
            } else {
                banner.hide();
            }
        }

        // Disable write UI in read-only (backend already enforces via processors)
        i18n.setWriteUiEnabled(writeAllowed);
    };

    /**
     * Enable/disable manager write UI controls.
     * This does not replace backend gating; it is only UX.
     */
    i18n.setWriteUiEnabled = function (enabled) {
        enabled = (enabled === true);

        // Disable Import buttons
        var importForm = Ext.getCmp('i18n-import-form');
        if (importForm && importForm.buttons) {
            Ext.each(importForm.buttons, function (b) {
                if (!b) return;
                // Import and Preview are both considered "write-ish" operations; keep preview enabled if you want.
                // Current policy: disable both when readonly.
                b.setDisabled(!enabled);
            });
        }

        // Disable Settings save if panel exposes a save button (depends on your settings panel implementation)
        var settingsPanel = Ext.getCmp('i18n-settings-panel');
        if (settingsPanel && settingsPanel.getForm && settingsPanel.buttons) {
            Ext.each(settingsPanel.buttons, function (b) {
                if (!b) return;
                b.setDisabled(!enabled);
            });
        }

        // Keys grid: disable topbar actions if possible (grid implementation-dependent)
        var grid = Ext.getCmp('i18n-grid-keys');
        if (grid && grid.getTopToolbar) {
            var tb = grid.getTopToolbar();
            if (tb && tb.items) {
                tb.items.each(function (item) {
                    if (!item) return;
                    item.setDisabled(!enabled);
                });
            }
        }

        // Also store flag for other scripts (keys.grid.js etc.)
        i18n.license = i18n.license || {};
        i18n.license.writeAllowed = enabled;
    };

    /**
     * License activation window (JSON + SIG).
     */
    i18n.openLicenseActivateWindow = function () {
        if (Ext.getCmp('i18n-license-activate-win')) {
            Ext.getCmp('i18n-license-activate-win').show();
            return;
        }

        var w = new MODx.Window({
            id: 'i18n-license-activate-win',
            title: 'Activate i18n License',
            width: 760,
            autoHeight: true,
            modal: true,
            closeAction: 'close',
            url: i18n.config.connectorUrl,
            baseParams: {
                action: 'mgr/license/activate',
                namespace: i18n.config.namespace || 'i18n'
            },
            fields: [{
                xtype: 'textarea',
                fieldLabel: 'License JSON',
                name: 'license_json',
                anchor: '100%',
                height: 220,
                allowBlank: false
            }, {
                xtype: 'textfield',
                fieldLabel: 'License Signature (base64)',
                name: 'license_sig',
                anchor: '100%',
                allowBlank: false
            }],
            buttons: [{
                text: _('cancel'),
                handler: function () { w.close(); }
            }, {
                text: _('save'),
                cls: 'primary-button',
                handler: function () {
                    var f = w.fp && w.fp.getForm ? w.fp.getForm() : null;
                    if (!f || !f.isValid()) {
                        MODx.msg.alert(_('error'), 'Please provide license JSON and signature.');
                        return;
                    }
                    w.submit({
                        success: function () {
                            w.close();
                            MODx.msg.status({ message: 'License activated.', delay: 2 });
                            i18n.refreshLicenseStatus();
                        },
                        failure: function (fp, r) {
                            MODx.msg.alert(_('error'), (r && r.message) ? r.message : 'Activation failed.');
                        }
                    });
                }
            }]
        });

        w.show();
    };

    i18n.panel.Home.superclass.constructor.call(this, config);

    // Load status after UI is ready
    Ext.onReady(function () {
        i18n.refreshLicenseStatus();
    });
};

Ext.extend(i18n.panel.Home, MODx.Panel);
Ext.reg('i18n-panel-home', i18n.panel.Home);
