(() => {
  const cfg = window.I18N_MGR_TOOLS || {};
  if (!cfg.connectorUrl) return;

  // Special "auto language" option:
  // - UI label: i18n.lex('general','General')
  // - Snippet value: [[!i18nGetLang]]
  const LANG_AUTO_VALUE = '__AUTO__';
  const LANG_AUTO_LABEL =
    (window.i18n && typeof i18n.lex === 'function')
      ? i18n.lex('general', 'General')
      : 'General';

  const LS_KEY_POS = 'i18n_mgr_btn_pos_v1';
  const LS_KEY_STATE = 'i18n_mgr_state_v1';

  let btn, panel, resultsWrap, inputEl, langEl, actionBtn, spinnerEl;
  let isOpen = false;
  let dragging = false;
  let dragOffset = { x: 0, y: 0 };
  let lastFocusable = null;
  let abortCtrl = null;
  let debounceT = null;
  let cache = new Map(); // query -> results

  document.addEventListener(
    'focusin',
    (e) => {
      const el = e.target;
      if (!el) return;

      // Ignore focus changes inside our widget
      if (isWidgetElement(el)) return;

      // Track last focused editable element (best-effort)
      if (
        el instanceof HTMLInputElement ||
        el instanceof HTMLTextAreaElement ||
        el.isContentEditable
      ) {
        // Optional: ignore readonly/disabled
        if (el instanceof HTMLInputElement || el instanceof HTMLTextAreaElement) {
          if (el.disabled || el.readOnly) return;
        }

        lastFocusable = el;
      }
    },
    true
  );

  function isWidgetElement(el) {
    if (!el) return false;
    // until created, btn/panel can be undefined
    if (btn && btn.contains(el)) return true;
    if (panel && panel.contains(el)) return true;

    // extra safety: by id prefix
    if (typeof el.id === 'string' && el.id.startsWith('i18n-mgr-')) return true;
    return false;
  }

  function safeJsonParse(s, fallback) {
    try {
      if (!s) return fallback;
      const v = JSON.parse(s);
      // JSON can be "null" => v === null, treat as invalid
      if (v === null || typeof v !== 'object') return fallback;
      return v;
    } catch {
      return fallback;
    }
  }

  function clampPos(x, y, snap = null) {
    const pad = 10;
    const w = 56, h = 56;

    const maxX = window.innerWidth - w - pad;
    const maxY = window.innerHeight - h - pad;

    let nx = Math.min(Math.max(pad, x), maxX);
    let ny = Math.min(Math.max(pad, y), maxY);

    if (snap && typeof snap === 'object') {
      if (snap.snapX === 'left') nx = pad;
      if (snap.snapX === 'right') nx = maxX;
      if (snap.snapY === 'top') ny = pad;
      if (snap.snapY === 'bottom') ny = maxY;
    }

    return { x: nx, y: ny };
  }

  function computeSnap(x, y) {
    const pad = 10;
    const w = 56, h = 56;
    const snapDist = 18; // "safe zone" in px

    const maxX = window.innerWidth - w - pad;
    const maxY = window.innerHeight - h - pad;

    let snapX = null;
    let snapY = null;

    if (x <= pad + snapDist) snapX = 'left';
    else if (x >= maxX - snapDist) snapX = 'right';

    if (y <= pad + snapDist) snapY = 'top';
    else if (y >= maxY - snapDist) snapY = 'bottom';

    return { snapX, snapY };
  }

  function loadBtnPos() {
    const pos = safeJsonParse(localStorage.getItem(LS_KEY_POS), null);

    // default position: bottom-right
    if (!pos || typeof pos.x !== 'number' || typeof pos.y !== 'number') {
      return { x: window.innerWidth - 76, y: window.innerHeight - 96, snapX: null, snapY: null };
    }

    const snap = { snapX: pos.snapX || null, snapY: pos.snapY || null };
    const p = clampPos(pos.x, pos.y, snap);
    return { ...p, ...snap };
  }

  function saveBtnPos(x, y, snapX = null, snapY = null) {
    localStorage.setItem(LS_KEY_POS, JSON.stringify({ x, y, snapX, snapY }));
  }

  function loadState() {
    const st = safeJsonParse(localStorage.getItem(LS_KEY_STATE), {});
    return {
      q: typeof st.q === 'string' ? st.q : '',
      // Default to "General" auto-language option
      lang: typeof st.lang === 'string' ? st.lang : LANG_AUTO_VALUE,
    };
  }

  function saveState() {
    if (!inputEl || !langEl) return;
    const st = { q: inputEl.value, lang: langEl.value };
    localStorage.setItem(LS_KEY_STATE, JSON.stringify(st));
  }

  function createBtn() {
    btn = document.createElement('button');
    btn.id = 'i18n-mgr-btn';
    btn.type = 'button';
    btn.innerHTML = `
      <i class="icon-language icon"></i>
      <div class="i18n-label">i18n</div>
    `;

    const pos = loadBtnPos();
    btn.style.left = pos.x + 'px';
    btn.style.top = pos.y + 'px';
    btn.dataset.snapX = pos.snapX || '';
    btn.dataset.snapY = pos.snapY || '';

    btn.addEventListener('pointerdown', (e) => {
      if (e.button !== 0) return;

      dragging = false;
      const r = btn.getBoundingClientRect();
      dragOffset.x = e.clientX - r.left;
      dragOffset.y = e.clientY - r.top;

      btn.setPointerCapture(e.pointerId);

      const onMove = (ev) => {
        dragging = true;

        const rawX = ev.clientX - dragOffset.x;
        const rawY = ev.clientY - dragOffset.y;

        // First clamp, then decide snapping
        let p = clampPos(rawX, rawY, null);
        const snap = computeSnap(p.x, p.y);

        // Apply snap if detected
        p = clampPos(p.x, p.y, snap);

        btn.style.left = p.x + 'px';
        btn.style.top = p.y + 'px';

        btn.dataset.snapX = snap.snapX || '';
        btn.dataset.snapY = snap.snapY || '';

        saveBtnPos(p.x, p.y, snap.snapX, snap.snapY);

        if (isOpen) positionPanel();
      };

      const onUp = () => {
        btn.removeEventListener('pointermove', onMove);
        btn.removeEventListener('pointerup', onUp);
        btn.removeEventListener('pointercancel', onUp);

        if (!dragging) togglePanel();
        dragging = false;
      };

      btn.addEventListener('pointermove', onMove);
      btn.addEventListener('pointerup', onUp);
      btn.addEventListener('pointercancel', onUp);
    });

    document.body.appendChild(btn);
  }

  function createPanel() {
    panel = document.createElement('div');
    panel.id = 'i18n-mgr-panel';
    panel.innerHTML = `
      <div class="row">
        <div class="spinner" id="i18n-mgr-spinner" aria-hidden="true"></div>
        <input id="i18n-mgr-input" type="text" placeholder="Search key / description / value..." autocomplete="off" />
        <select id="i18n-mgr-lang"></select>
        <button id="i18n-mgr-insert" type="button">` + i18n.lex('insert', 'Insert') + `</button>
      </div>
      <div id="i18n-mgr-results">
        <table>
          <thead>
            <tr><th style="width:45%">`+ i18n.lex('i18n_tab_keys', 'Keys') +`</th><th>` + i18n.lex('i18n_description', 'Description') + `</th></tr>
          </thead>
          <tbody></tbody>
        </table>
      </div>
      <div id="i18n-mgr-toast" role="status" aria-live="polite"></div>
    `;
    document.body.appendChild(panel);

    spinnerEl = panel.querySelector('#i18n-mgr-spinner');
    inputEl = panel.querySelector('#i18n-mgr-input');
    langEl = panel.querySelector('#i18n-mgr-lang');
    actionBtn = panel.querySelector('#i18n-mgr-insert');
    resultsWrap = panel.querySelector('#i18n-mgr-results');

    inputEl.addEventListener('blur', () => {
      setTimeout(() => {
        if (document.activeElement !== inputEl) hideResults();
      }, 120);
      saveState();
    });

    inputEl.addEventListener('focus', () => {
      const q = inputEl.value.trim();
      if (q && cache.has(q)) renderResults(cache.get(q));
    });

    inputEl.addEventListener('input', () => {
      saveState();
      scheduleSearch();
    });

    inputEl.addEventListener('keydown', (e) => {
      if (e.key === 'Enter') {
        e.preventDefault();
        doInsertOrCopy();
      }
    });

    actionBtn.addEventListener('click', () => doInsertOrCopy());

    // Click outside closes panel
    document.addEventListener(
      'mousedown',
      (e) => {
        if (!isOpen) return;
        if (panel.contains(e.target) || btn.contains(e.target)) return;
        closePanel();
      },
      true
    );

    window.addEventListener('resize', () => {
      // keep btn snapped if it was snapped
      const snap = { snapX: btn?.dataset?.snapX || null, snapY: btn?.dataset?.snapY || null };
      const cur = btn.getBoundingClientRect();
      const p = clampPos(cur.left, cur.top, snap);

      btn.style.left = p.x + 'px';
      btn.style.top = p.y + 'px';
      saveBtnPos(p.x, p.y, snap.snapX || null, snap.snapY || null);

      if (isOpen) {
        positionPanel();
        if (resultsWrap.style.display === 'block') positionResults();
      }
    });
  }

  function setSpinner(on) {
    if (!spinnerEl) return;
    spinnerEl.classList.toggle('is-on', !!on);
  }

  function positionPanel() {
    const br = btn.getBoundingClientRect();
    const pw = panel.offsetWidth || 520;
    const ph = panel.offsetHeight || 80;

    const pad = 10;
    const spaceLeft = br.left;
    const spaceRight = window.innerWidth - br.right;

    const openToRight = spaceRight >= spaceLeft;

    let left = openToRight ? br.right + pad : br.left - pw - pad;
    let top = br.top + br.height / 2 - ph / 2;

    left = Math.min(Math.max(pad, left), window.innerWidth - pw - pad);
    top = Math.min(Math.max(pad, top), window.innerHeight - ph - pad);

    panel.style.left = left + 'px';
    panel.style.top = top + 'px';

    // If results already visible, keep them oriented correctly
    if (resultsWrap && resultsWrap.style.display === 'block') {
      positionResults();
    }
  }

  function showPanel() {
    // Capture current target before widget steals focus
    const active = document.activeElement;
    if (active && !isWidgetElement(active)) {
      if (
        active instanceof HTMLInputElement ||
        active instanceof HTMLTextAreaElement ||
        active.isContentEditable
      ) {
        lastFocusable = active;
      }
    }

    isOpen = true;
    panel.style.display = 'block';
    positionPanel();

    const st = loadState();
    inputEl.value = st.q || '';
    langEl.value = st.lang || LANG_AUTO_VALUE;

    setTimeout(() => inputEl.focus(), 0);

    if (inputEl.value.trim()) scheduleSearch();
  }

  function closePanel() {
    isOpen = false;
    panel.style.display = 'none';
    hideResults();
    setSpinner(false);

    if (abortCtrl) abortCtrl.abort();
    abortCtrl = null;

    if (debounceT) clearTimeout(debounceT);
    debounceT = null;

    saveState();
  }

  function togglePanel() {
    if (isOpen) closePanel();
    else showPanel();
  }

  // Results should open above or below depending on available space
  function positionResults() {
    const rowEl = panel.querySelector('.row');
    if (!rowEl || !resultsWrap) return;

    const rowRect = rowEl.getBoundingClientRect();
    const panelRect = panel.getBoundingClientRect();

    const spaceAbove = rowRect.top - 10;
    const spaceBelow = window.innerHeight - rowRect.bottom - 10;

    const openUp = spaceAbove > spaceBelow;

    if (openUp) {
      // Attach list above the row
      resultsWrap.style.top = '';
      // bottom relative to panel: distance from row top to panel top plus a small gap
      const bottomPx = panelRect.height - (rowRect.top - panelRect.top) + 14;
      resultsWrap.style.bottom = bottomPx + 'px';
    } else {
      // Attach list below the row
      resultsWrap.style.bottom = '';
      resultsWrap.style.top = rowEl.offsetTop + rowEl.offsetHeight + 14 + 'px';
    }
  }

  function hideResults() {
    if (!resultsWrap) return;
    resultsWrap.style.display = 'none';
    const tbody = resultsWrap.querySelector('tbody');
    if (tbody) tbody.innerHTML = '';
  }

  function renderResults(rows) {
    const tbody = resultsWrap.querySelector('tbody');
    if (!tbody) return;

    tbody.innerHTML = '';

    if (!rows || !rows.length) {
      hideResults();
      return;
    }

    // UI shows 5 rows max (DB returns up to 20)
    for (const r of rows.slice(0, 5)) {
      const tr = document.createElement('tr');
      const key = (r.key || '').toString();
      const desc = (r.description || '').toString();

      tr.innerHTML = `<td class="k"></td><td class="d"></td>`;
      tr.querySelector('.k').textContent = key;
      tr.querySelector('.d').textContent = desc;

      // Prevent input blur on click
      tr.addEventListener('mousedown', (e) => e.preventDefault());

      tr.addEventListener('click', () => {
        inputEl.value = key;
        saveState();
        hideResults();
        inputEl.focus();
      });

      tbody.appendChild(tr);
    }

    positionResults();
    resultsWrap.style.display = 'block';
  }

  function shouldSearch(q) {
    if (!q) return false;
    if (q.length > 128) return true;
    if (q.endsWith('.') && q.length >= 1) return true;
    return q.length >= 2;
  }

  function scheduleSearch() {
    if (debounceT) clearTimeout(debounceT);
    debounceT = setTimeout(doSearch, 300);
  }

  async function doSearch() {
    const q = inputEl.value.trim();

    if (document.activeElement !== inputEl) return;

    if (!shouldSearch(q)) {
      setSpinner(false);
      hideResults();
      return;
    }

    if (cache.has(q)) {
      setSpinner(false);
      renderResults(cache.get(q));
      return;
    }

    if (abortCtrl) abortCtrl.abort();
    abortCtrl = new AbortController();

    setSpinner(true);
    hideResults();

    try {
      const url = new URL(cfg.connectorUrl, window.location.origin);
      url.searchParams.set('action', 'mgr/search');
      url.searchParams.set('namespace', cfg.namespace || 'i18n');

      const form = new FormData();
      form.set('q', q);
      form.set('limit', '20');

      const modAuth =
        window.MODx && (MODx.siteId || MODx.config?.site_id)
          ? (MODx.siteId || MODx.config.site_id)
          : '';

      if (modAuth) {
        url.searchParams.set('HTTP_MODAUTH', modAuth);
      }

      const res = await fetch(url.toString(), {
        method: 'POST',
        body: form,
        signal: abortCtrl.signal,
        headers: {
          'X-Requested-With': 'XMLHttpRequest',
          ...(modAuth ? { modAuth } : {}),
        },
        credentials: 'same-origin',
      });

      const json = await res.json();

      // Ignore stale responses
      if (inputEl.value.trim() !== q) return;
      if (document.activeElement !== inputEl) return;

      if (!json || json.success !== true) {
        setSpinner(false);
        hideResults();
        return;
      }

      const rows = json.object && json.object.results ? json.object.results : [];
      cache.set(q, rows);

      setSpinner(false);
      renderResults(rows);
    } catch (e) {
      if (e && e.name === 'AbortError') return;
      setSpinner(false);
      hideResults();
    }
  }

  function buildSnippet(fullKey, lang) {
    const safeKey = (fullKey || '').trim();

    // If user selected "General" (auto), insert i18nGetLang call.
    // Otherwise use explicit language code.
    let safeLang;
    if (lang === LANG_AUTO_VALUE) {
      safeLang = '[[!i18nGetLang]]';
    } else {
      safeLang = (lang || '').trim();
    }

    return `[[!i18nKey? &key=\`${safeKey}\` &lang=\`${safeLang}\`]]`;
  }

  async function copyToClipboard(text) {
    try {
      await navigator.clipboard.writeText(text);
      return true;
    } catch {
      const ta = document.createElement('textarea');
      ta.value = text;
      ta.style.position = 'fixed';
      ta.style.left = '-9999px';
      document.body.appendChild(ta);
      ta.select();
      const ok = document.execCommand('copy');
      document.body.removeChild(ta);
      return ok;
    }
  }

  function insertIntoElement(el, text) {
    if (!el) return false;

    if (el.isContentEditable) {
      el.focus();
      document.execCommand('insertText', false, text);
      return true;
    }

    if (el instanceof HTMLInputElement || el instanceof HTMLTextAreaElement) {
      const start = el.selectionStart ?? el.value.length;
      const end = el.selectionEnd ?? el.value.length;

      el.value = el.value.slice(0, start) + text + el.value.slice(end);

      const caret = start + text.length;
      el.setSelectionRange(caret, caret);

      el.dispatchEvent(new Event('input', { bubbles: true }));
      el.focus();
      return true;
    }

    return false;
  }

  async function doInsertOrCopy() {
    const key = inputEl.value.trim();
    if (!key) return;

    const snippet = buildSnippet(key, langEl.value);

    const inserted = insertIntoElement(lastFocusable, snippet);
    if (inserted) {
      showToast(
        i18n.lex('i18n_inserted_into_field', 'Inserted into the last focused field.'),
        'ok'
      );
      actionBtn.textContent = 'Inserted';
      setTimeout(() => (actionBtn.textContent = i18n.lex('insert', 'Insert')), 900);
      return;
    }

    const copied = await copyToClipboard(snippet);
    if (copied) {
      showToast(i18n.lex('i18n_copy_clipboard', 'Copied to clipboard (no editable field detected).'), 'ok');
    } else {
      showToast(i18n.lex('i18n_copy_failed', 'Copy failed. Please copy manually.'), 'warn');
    }

    actionBtn.textContent = 'Copy';
    setTimeout(() => (actionBtn.textContent = i18n.lex('insert', 'Insert')), 1200);
  }

  function fillLangSelectFromCfg() {
    langEl.innerHTML = '';

    // Always add "General" (auto) as the first option
    {
      const opt = document.createElement('option');
      opt.value = LANG_AUTO_VALUE;
      opt.textContent = LANG_AUTO_LABEL;
      langEl.appendChild(opt);
    }

    const list = Array.isArray(cfg.langs) ? cfg.langs : null;

    if (list && list.length) {
      for (const item of list) {
        // Skip legacy placeholder from backend if present
        if (!item || item.value === '—') continue;

        const opt = document.createElement('option');
        opt.value = item.value;
        opt.textContent = item.label;
        langEl.appendChild(opt);
      }
    } else {
      // Minimal fallback: keep "General" only
    }

    const st = loadState();
    if (st.lang) {
      if ([...langEl.options].some(o => o.value === st.lang)) {
        langEl.value = st.lang;
      } else {
        langEl.selectedIndex = 0;
      }
    } else {
      langEl.selectedIndex = 0;
    }
  }

  let toastT = null;

  function showToast(text, kind = 'ok') {
    const el = panel.querySelector('#i18n-mgr-toast');
    if (!el) return;

    el.textContent = text;
    el.classList.remove('is-ok', 'is-warn', 'is-show');
    el.classList.add(kind === 'warn' ? 'is-warn' : 'is-ok', 'is-show');

    if (toastT) clearTimeout(toastT);
    toastT = setTimeout(() => {
      el.classList.remove('is-show', 'is-ok', 'is-warn');
    }, 1400);
  }

  function init() {
    createBtn();
    createPanel();
    fillLangSelectFromCfg();

    window.I18N_MGR_TOOLS_API = window.I18N_MGR_TOOLS_API || {};
    window.I18N_MGR_TOOLS_API.update = function (payload) {
      if (!payload) return;

      // update global config
      if (payload.list && Array.isArray(payload.list)) {
        cfg.langs = payload.list;
      }

      // update UI if already initialized
      if (langEl) {
        const prev = langEl.value;

        langEl.innerHTML = '';

        // Always add "General" (auto) first
        {
          const opt = document.createElement('option');
          opt.value = LANG_AUTO_VALUE;
          opt.textContent = LANG_AUTO_LABEL;
          langEl.appendChild(opt);
        }

        if (cfg.langs && Array.isArray(cfg.langs)) {
          for (const item of cfg.langs) {
            if (!item || item.value === '—') continue;

            const opt = document.createElement('option');
            opt.value = item.value;
            opt.textContent = item.label;
            langEl.appendChild(opt);
          }
        }

        // keep selection if possible
        if ([...langEl.options].some(o => o.value === prev)) {
          langEl.value = prev;
        } else {
          langEl.value = LANG_AUTO_VALUE;
        }
      }
    };
  }

  window.addEventListener('i18n:settingsSaved', (e) => {
    const payload = e && e.detail ? e.detail : null;
    if (window.I18N_MGR_TOOLS_API && typeof window.I18N_MGR_TOOLS_API.update === 'function') {
      window.I18N_MGR_TOOLS_API.update(payload);
    }
  });

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }
})();
