/**
 * i18n main component namespace for MODX manager.
 *
 * This file defines the main ExtJS component, creates a singleton instance,
 * and exposes a shared config + helper methods for the rest of the JS code.
 */

var i18n = function (config) {
    config = config || {};
    i18n.superclass.constructor.call(this, config);
};

Ext.extend(i18n, Ext.Component, {
    page: {},
    window: {},
    grid: {},
    panel: {},
    config: {},
    combo: {},
    view: {}
});

Ext.reg('i18n', i18n);
i18n = new i18n();

i18n.config = i18n.config || {};
i18n.config.connectorUrl = MODx.config.assets_url + 'components/i18n/connector.php';
i18n.config.namespace = 'i18n';

/**
 * Safe lexicon lookup with a fallback.
 */
i18n.lex = function (key, def) {
    if (window.I18N_LEXICON && window.I18N_LEXICON[key]) {
        return window.I18N_LEXICON[key];
    }

    if (typeof _ === 'function') {
        var value = _(key);
        if (value && value !== key) {
            return value;
        }
    }

    return def || key;
};

/**
 * Extract "object" from MODX Ajax response (supports both shapes).
 */
i18n._extractObject = function (r) {
    if (!r) return null;
    if (r.object && typeof r.object === 'object') return r.object;
    if (r.result && r.result.object && typeof r.result.object === 'object') return r.result.object;
    return null;
};

/**
 * Store license object (raw server payload) and notify listeners.
 */
i18n._applyLicenseObject = function (obj) {
    if (obj && typeof obj === 'object') {
        i18n.config.license = obj;
    }
    i18n.emitLicenseUpdated();
};

/**
 * Normalize licensing status into a stable shape used by UI.
 * Supports both snake_case (server) and camelCase (older injected variants).
 */
i18n.license = function () {
    var lic = (i18n.config && i18n.config.license) ? i18n.config.license : null;

    var mode = (lic && lic.mode) ? String(lic.mode) : 'readonly';

    var writeAllowed = false;
    if (lic) {
        if (lic.writeAllowed === true || lic.writeAllowed === 1 || lic.writeAllowed === '1') writeAllowed = true;
        if (lic.write_allowed === true || lic.write_allowed === 1 || lic.write_allowed === '1') writeAllowed = true;
    }

    var trialLeft = null;
    if (lic) {
        if (typeof lic.trialLeftSeconds === 'number') trialLeft = lic.trialLeftSeconds;
        if (typeof lic.trial_left_seconds === 'number') trialLeft = lic.trial_left_seconds;
    }

    var graceLeft = null;
    if (lic) {
        if (typeof lic.graceLeftSeconds === 'number') graceLeft = lic.graceLeftSeconds;
        if (typeof lic.grace_left_seconds === 'number') graceLeft = lic.grace_left_seconds;
    }

    return {
        mode: mode,
        writeAllowed: writeAllowed,
        trialLeftSeconds: trialLeft,
        graceLeftSeconds: graceLeft,
        reason: (lic && lic.reason) ? String(lic.reason) : '',
        fingerprint: (lic && lic.fingerprint) ? String(lic.fingerprint) : ''
    };
};

i18n.isWriteAllowed = function () {
    return i18n.license().writeAllowed === true;
};

i18n.isReadOnly = function () {
    return i18n.license().mode === 'readonly' || i18n.isWriteAllowed() === false;
};

i18n.formatDuration = function (seconds) {
    seconds = parseInt(seconds, 10);
    if (!seconds || seconds <= 0) return '0';

    var d = Math.floor(seconds / 86400);
    var h = Math.floor((seconds % 86400) / 3600);
    var m = Math.floor((seconds % 3600) / 60);

    var parts = [];
    if (d) parts.push(d + 'd');
    if (h) parts.push(h + 'h');
    if (!d && m) parts.push(m + 'm');

    return parts.join(' ');
};

i18n.showLicenseNotice = function (opts) {
    opts = opts || {};
    var lic = i18n.license();

    if (opts.force !== true) {
        if (lic.mode === 'licensed' || lic.mode === 'trial') {
            return;
        }
    }

    if (lic.mode === 'grace') {
        var left = (typeof lic.graceLeftSeconds === 'number') ? i18n.formatDuration(lic.graceLeftSeconds) : '';
        MODx.msg.alert(
            i18n.lex('i18n_license_grace_title', 'License check-in required'),
            i18n.lex(
                'i18n_license_grace_text',
                'Trial period has ended. Please activate license (check-in) to avoid switching to read-only mode.'
            ) + (left ? ('<br><br><b>' + i18n.lex('i18n_time_left', 'Time left') + ':</b> ' + Ext.util.Format.htmlEncode(left)) : '')
        );
        return;
    }

    if (lic.mode === 'readonly') {
        MODx.msg.alert(
            i18n.lex('i18n_readonly_title', 'Read-only mode'),
            i18n.lex(
                'i18n_readonly_text',
                'i18n is running in read-only mode. A valid license is required to create, update, delete, import, export and rebuild usage.'
            )
        );
    }
};

i18n.guardWrite = function (fn, scope) {
    if (i18n.isWriteAllowed()) {
        if (typeof fn === 'function') {
            fn.call(scope || this);
        }
        return true;
    }

    i18n.showLicenseNotice({ force: true });
    return false;
};

/**
 * Emit event so other UI parts can re-gate themselves.
 */
i18n.emitLicenseUpdated = function () {
    try {
        window.dispatchEvent(new CustomEvent('i18n:licenseUpdated', { detail: i18n.license() }));
    } catch (e) {}
};

/**
 * Refresh license status from server (source of truth).
 */
i18n.refreshLicenseStatus = function (cb) {
    MODx.Ajax.request({
        url: i18n.config.connectorUrl,
        params: {
            action: 'mgr/license/status',
            namespace: i18n.config.namespace || 'i18n'
        },
        listeners: {
            success: {
                fn: function (r) {
                    var obj = i18n._extractObject(r);
                    if (obj) {
                        i18n._applyLicenseObject(obj);
                    } else {
                        i18n.emitLicenseUpdated();
                    }
                    if (typeof cb === 'function') cb(true, r);
                },
                scope: this
            },
            failure: {
                fn: function (r) {
                    i18n.emitLicenseUpdated();
                    if (typeof cb === 'function') cb(false, r);
                },
                scope: this
            }
        }
    });
};

/**
 * Activate license (offline) and refresh UI state.
 */
i18n.activateLicense = function (licenseJson, licenseSig, cb) {
    licenseJson = String(licenseJson || '').trim();
    licenseSig  = String(licenseSig || '').trim();

    if (!licenseJson || !licenseSig) {
        MODx.msg.alert(i18n.lex('error', 'Error'), 'Missing license JSON or signature.');
        return;
    }

    MODx.Ajax.request({
        url: i18n.config.connectorUrl,
        params: {
            action: 'mgr/license/activate',
            namespace: i18n.config.namespace || 'i18n',
            license_json: licenseJson,
            license_sig: licenseSig
        },
        listeners: {
            success: {
                fn: function (r) {
                    // After activation we always refresh status from server
                    i18n.refreshLicenseStatus(function () {
                        if (typeof cb === 'function') cb(true, r);
                    });
                },
                scope: this
            },
            failure: {
                fn: function (r) {
                    i18n.emitLicenseUpdated();
                    if (typeof cb === 'function') cb(false, r);
                },
                scope: this
            }
        }
    });
};

/**
 * Online check-in (endpoint).
 * Calls mgr/license/checkin and updates license state from response.
 */
i18n.checkInLicense = function (cb) {
    MODx.Ajax.request({
        url: i18n.config.connectorUrl,
        params: {
            action: 'mgr/license/checkin',
            namespace: i18n.config.namespace || 'i18n'
        },
        listeners: {
            success: {
                fn: function (r) {
                    // Processor returns updated status in object
                    var obj = i18n._extractObject(r);
                    if (obj) {
                        i18n._applyLicenseObject(obj);
                    } else {
                        // Fallback: refresh from server
                        i18n.refreshLicenseStatus(function () {
                            if (typeof cb === 'function') cb(true, r);
                        });
                        return;
                    }

                    if (typeof cb === 'function') cb(true, r);
                },
                scope: this
            },
            failure: {
                fn: function (r) {
                    // Even on failure, status might have changed (grace ticking, etc.)
                    i18n.refreshLicenseStatus(function () {
                        if (typeof cb === 'function') cb(false, r);
                    });
                },
                scope: this
            }
        }
    });
};

// Refresh once on manager load to avoid stale injected data
Ext.onReady(function () {
    i18n.refreshLicenseStatus();
});
