/**
 * Window for updating an existing i18n key.
 *
 * Behavior:
 *  - Key is always read-only
 *  - Group is editable only when "Unlock renaming" is enabled
 *  - On save:
 *      - If unlock is OFF => mgr/update
 *      - If unlock is ON:
 *          - group empty OR same as original => mgr/update (no rename)
 *          - group changed => mgr/rekey (rename + bulk replace old_key -> new_key)
 *
 * @class i18n.window.KeyUpdate
 * @extends MODx.Window
 * @xtype i18n-window-key-update
 */

i18n.window = i18n.window || {};

i18n.window.KeyUpdate = function (config) {
    config = config || {};
    this.record = config.record || {};

    Ext.applyIf(config, {
        title: i18n.lex('i18n_update_key', 'Edit key'),
        id: 'i18n-window-key-update',

        width: 700,
        height: 600,

        layout: 'fit',
        modal: true,
        closeAction: 'close',
        resizable: true,

        cls: 'modx-window',

        url: i18n.config.connectorUrl,
        baseParams: {
            // NOTE: actual action is chosen dynamically in submitForm()
            action: 'mgr/update',
            namespace: i18n.config.namespace || 'i18n',
            id: (config.record && config.record.id) ? config.record.id : 0
        },

        items: [{
            xtype: 'form',
            layout: 'form',
            autoScroll: true,
            border: false,
            bodyStyle: 'padding: 10px;',
            labelAlign: 'top',
            items: this.getFields(config)
        }],

        buttons: [{
            text: _('save'),
            cls: 'primary-button',
            handler: function () {
                this.submitForm();
            },
            scope: this
        },{
            text: _('cancel'),
            handler: function () { this.close(); },
            scope: this
        }]
    });

    i18n.window.KeyUpdate.superclass.constructor.call(this, config);
};

Ext.extend(i18n.window.KeyUpdate, MODx.Window, {

    /**
     * Defines form fields for editing key and translations.
     */
    getFields: function () {
        var fields = [];

        // Hidden ID
        fields.push({
            xtype: 'hidden',
            name: 'id'
        });

        // Full key (read-only)
        fields.push({
            xtype: 'displayfield',
            fieldLabel: i18n.lex('i18n_key', 'Key'),
            name: 'key',
            anchor: '100%'
        });

        // Unlock checkbox controls group editability
        fields.push({
            xtype: 'xcheckbox',
            id: 'i18n-update-unlock-renaming',
            name: 'unlock_renaming',
            boxLabel: i18n.lex('i18n_unlock_renaming', 'Unlock renaming'),
            checked: false,
            listeners: {
                check: function (cb, checked) {
                    var groupField = Ext.getCmp('i18n-update-group');
                    if (!groupField) return;

                    groupField.setDisabled(!checked);

                    // When locking back, restore original value from loaded record
                    if (!checked) {
                        var rec = this.record || {};
                        groupField.setValue(rec.group || '');
                    }
                },
                scope: this
            }
        });

        // Group becomes editable only when unlocked
        fields.push({
            xtype: 'textfield',
            fieldLabel: i18n.lex('i18n_group', 'Group'),
            id: 'i18n-update-group',
			regex: /^[A-Za-z0-9_]*$/,
			regexText: i18n.lex('i18n_group_regex', 'Only Latin letters, numbers and the special character "_"'),
            name: 'group',
            anchor: '99%',
            disabled: true
        });

        // Description
        fields.push({
            xtype: 'textarea',
            fieldLabel: i18n.lex('i18n_description', 'Description'),
            name: 'description',
            anchor: '99%',
            height: 70
        });

        var langs = this.getLanguagesForTabs();
        var defaultLang = i18n.config.defaultLang || 'en';

        if (langs.length) {
            fields.push({
                xtype: 'label',
				anchor: '99%',
                html: '<hr style="margin: 10px 0;" />'
            });

            fields.push({
                xtype: 'modx-tabs',
                plain: true,
                anchor: '99%',
                deferredRender: false,
                defaults: {
                    border: false,
                    autoHeight: true,
                    layout: 'form',
                    bodyStyle: 'padding: 10px'
                },
                items: this.buildLangTabs(langs, defaultLang)
            });
        }

        return fields;
    },

    /**
     * Build language list for tabs.
     */
    getLanguagesForTabs: function () {
        var rawLangs = [];
        var cfg = i18n.config || {};

        if (cfg.allowedLangs) {
            if (Ext.isArray(cfg.allowedLangs)) {
                rawLangs = cfg.allowedLangs.slice();
            } else {
                rawLangs = String(cfg.allowedLangs).split(',');
            }
        }

        var def = cfg.defaultLang || 'en';

        // Normalize and remove duplicates (preserve order)
        var map = {};
        var langs = [];
        Ext.each(rawLangs, function (l) {
            var code = String(l || '').trim();
            if (!code || map[code]) return;
            map[code] = true;
            langs.push(code);
        });

        // Ensure default is present
        if (!map[def]) {
            langs.push(def);
        }

        // Move default to the first position
        var idx = langs.indexOf(def);
        if (idx > 0) {
            langs.splice(idx, 1);
            langs.unshift(def);
        } else if (idx === -1) {
            langs.unshift(def);
        }

        return langs;
    },

    /**
     * Build tabs for each language.
     */
    buildLangTabs: function (langs, defaultLang) {
        var items = [];

        Ext.each(langs, function (lang) {
            items.push({
                title: (lang === defaultLang ? lang + ' (default)' : lang),
                items: [{
                    xtype: 'textarea',
                    anchor: '100%',
                    name: 'value_' + lang,
                    height: 180,
                    hideLabel: true
                }]
            });
        });

        return items;
    },

    /**
     * Fill the form with record data when window is shown.
     */
    show: function () {
        i18n.window.KeyUpdate.superclass.show.call(this);

        if (!this.record) return;

        var formPanel = this.items.get(0);
        if (!formPanel || !formPanel.getForm) return;

        var form   = formPanel.getForm();
        var record = this.record;
        var values = record.values || {};
        var langs  = this.getLanguagesForTabs();

        form.setValues({
            id: record.id,
            key: record.key,
            group: record.group || '',
            description: record.description || ''
        });

        // IMPORTANT: group is locked by default
        var unlock = Ext.getCmp('i18n-update-unlock-renaming');
        var groupField = Ext.getCmp('i18n-update-group');
        if (unlock) unlock.setValue(false);
        if (groupField) groupField.setDisabled(true);

        // Language values: only for allowed+default languages
        Ext.each(langs, function (lang) {
            var field = form.findField('value_' + lang);
            if (!field) return;

            var v = values[lang] || '';
            field.setValue(v);
        });
    },

    /**
     * Custom submit handler.
     * Chooses processor based on unlock + group change:
     *  - mgr/update => normal update (description + values)
     *  - mgr/rekey  => rename (move prefix/group) + bulk replace content + update values
     */
    submitForm: function () {
        var formPanel = this.items.get(0);
        if (!formPanel || !formPanel.getForm) return;

        var form  = formPanel.getForm();
        var data  = form.getValues();
        var langs = this.getLanguagesForTabs();

        var record = this.record || {};
        var originalGroup = (record.group || '').trim();

        // Parse unlock flag robustly (Ext may return on/'1'/true)
        var unlock = false;
        if (typeof data.unlock_renaming !== 'undefined') {
            unlock = (data.unlock_renaming === true ||
                data.unlock_renaming === 1 ||
                data.unlock_renaming === '1' ||
                data.unlock_renaming === 'on');
        }

        var newGroup = (data.group || '').trim();

        // Decide which action to call
        var action = 'mgr/update';

        // Only attempt rename when unlocked AND group is non-empty AND changed
        if (unlock && newGroup !== '' && newGroup !== originalGroup) {
            action = 'mgr/rekey';
        }

        var params = {
            action: action,
            namespace: i18n.config.namespace || 'i18n'
        };

        // Key ID (required)
        params.id = data.id || record.id || 0;

        // Description (optional)
        params.description = data.description || '';

        // Always send unlock flag (server should enforce anyway)
        params.unlock_renaming = unlock ? 1 : 0;

        // Send group only when unlocked
        // If empty or same as original -> server will treat as no-op and keep as is
        if (unlock) {
            params.group = newGroup;
        }

        // When doing rekey, ask server to replace in content (safe defaults)
        if (action === 'mgr/rekey') {
            params.replace_content = 1;
            params.replace_resources = 1;
        }

        // Language values
        Ext.each(langs, function (lang) {
            var fieldName = 'value_' + lang;
            if (typeof data[fieldName] !== 'undefined') {
                params[fieldName] = data[fieldName];
            }
        });

        MODx.Ajax.request({
            url: i18n.config.connectorUrl,
            params: params,
            waitMsg: i18n.lex('saving', 'Saving...'),
            listeners: {
                success: {
                    fn: function (r) {
                        MODx.msg.status({
                            message: (r.result && r.result.message) ||
                                i18n.lex('i18n_key_updated', 'Key updated.'),
                            delay: 3
                        });

                        this.fireEvent('success', r);
                        this.close();
                    },
                    scope: this
                },
                failure: {
                    fn: function (r) {
                        MODx.msg.alert(
                            i18n.lex('error', 'Error'),
                            (r.result && r.result.message) ||
                                i18n.lex('i18n_update_error', 'Error while updating key.')
                        );
                    },
                    scope: this
                }
            }
        });
    }
});

Ext.reg('i18n-window-key-update', i18n.window.KeyUpdate);
