/**
 * Window for creating a new i18n key.
 *
 * @class i18n.window.KeyCreate
 * @extends MODx.Window
 * @xtype i18n-window-key-create
 */

i18n = window.i18n || {};
i18n.window = i18n.window || {};

i18n.window.KeyCreate = function (config) {
    config = config || {};

    Ext.applyIf(config, {
        title: i18n.lex('i18n_create_key', 'Create key'),
        id: 'i18n-window-key-create',
        width: 640,
        autoHeight: true,
        closeAction: 'hide',
        modal: true,
        resizable: false,
        maximizable: false,

        url: i18n.config.connectorUrl,
        baseParams: {
            action: 'mgr/create',
            namespace: i18n.config.namespace || 'i18n'
        },

        fields: this.getFields(config),

        keys: [{
            key: Ext.EventObject.ENTER,
            shift: false,
            fn: function () {
                this.submit();
            },
            scope: this
        }]
    });

    i18n.window.KeyCreate.superclass.constructor.call(this, config);
};

Ext.extend(i18n.window.KeyCreate, MODx.Window, {
    /**
     * Build fields for the create-key window.
     *
     * Group:
     *   Optional. If empty, backend will use "manual".
     *
     * Key:
     *   Optional. If empty, backend will generate a random suffix and build full key.
     *
     * Language values:
     *   One tab per language from i18n.config.languages / i18n.config.allowedLangs.
     *   Field names: <lang>, e.g. "en", "uk", "ru".
     */
    getFields: function (config) {
        var fields = [];

        // Optional key suffix
        fields.push({
            xtype: 'textfield',
            fieldLabel: i18n.lex('i18n_key', 'Key'),
            description: i18n.lex('i18n_key_desc', 'If empty, a random key will be generated'),
			regex: /^[A-Za-z0-9_]*$/,
			regexText: i18n.lex('i18n_group_regex', 'Only Latin letters, numbers and the special character "_"'),
            name: 'key',
            anchor: '100%',
            allowBlank: true
        });

        // Optional group
        fields.push({
            xtype: 'textfield',
            fieldLabel: i18n.lex('i18n_group', 'Group'),
            description: i18n.lex(
                'i18n_group_desc',
                'If the field is empty, the group specified in the settings will be used.'
            ),
			regex: /^[A-Za-z0-9_]*$/,
			regexText: i18n.lex('i18n_group_regex', 'Only Latin letters, numbers and the special character "_"'),
            name: 'group',
            anchor: '100%',
            allowBlank: true
        });

        // Optional description
        fields.push({
            xtype: 'textarea',
            fieldLabel: i18n.lex('i18n_description', 'Description'),
            name: 'description',
            anchor: '100%',
            allowBlank: true,
            grow: true,
            height: 60
        });

        // Language tabs
        var langs = this.getLanguagesForTabs();
        var defaultLang = i18n.config.defaultLang || 'en';

        if (langs.length) {
            fields.push({
                xtype: 'label',
                html: '<hr style="margin: 10px 0;" />'
            });

            fields.push({
                xtype: 'modx-tabs',
                plain: true,
                anchor: '100%',
                deferredRender: false,
                defaults: {
                    border: false,
                    autoHeight: true,
                    layout: 'form',
                    bodyStyle: 'padding: 10px'
                },
                items: this.buildLangTabs(langs, defaultLang)
            });
        }

        return fields;
    },

	/**
	 * Resolve languages for editor tabs from config.
	 * Guarantees at least one language and puts default language first.
	 *
	 * @returns {Array<string>}
	 */
	getLanguagesForTabs: function () {
		var cfg = i18n.config || {};
		var raw = [];

		if (Ext.isArray(cfg.languages)) {
			raw = cfg.languages.slice();
		} else if (Ext.isArray(cfg.allowedLangs)) {
			raw = cfg.allowedLangs.slice();
		} else if (typeof cfg.allowedLangs === 'string') {
			raw = cfg.allowedLangs.split(',');
		}

		var map = {};
		var langs = [];

		// Normalize and deduplicate, keep order
		Ext.each(raw, function (l) {
			var code = String(l || '').trim();
			if (!code || map[code]) return;
			map[code] = true;
			langs.push(code);
		});

		var defaultLang = cfg.defaultLang || 'en';

		// Ensure default exists
		if (!map[defaultLang]) {
			langs.push(defaultLang);
		}

		// Move default to first position
		var idx = langs.indexOf(defaultLang);
		if (idx > 0) {
			langs.splice(idx, 1);
			langs.unshift(defaultLang);
		} else if (idx === -1) {
			langs.unshift(defaultLang);
		}

		return langs;
	},

    /**
     * Build tab items for each language.
     *
     * @param {Array<string>} langs
     * @param {String} defaultLang
     * @returns {Array<Object>}
     */
    buildLangTabs: function (langs, defaultLang) {
        var items = [];

        Ext.each(langs, function (lang) {
            var title = lang;
            if (defaultLang && lang === defaultLang) {
                title = lang + ' (' + i18n.lex('i18n_default', 'default') + ')';
            }

            items.push({
                title: title,
                items: [{
                    xtype: 'textarea',
                    hideLabel: true,
                    name: lang,
                    anchor: '100%',
                    grow: true,
                    allowBlank: lang !== defaultLang,
                    emptyText: i18n.lex(
                        'i18n_value_for_lang',
                        'Text for language: '
                    ) + lang
                }]
            });
        });

        return items;
    }
});

Ext.reg('i18n-window-key-create', i18n.window.KeyCreate);
