/**
 * Grid for managing i18n keys.
 *
 * @class i18n.grid.Keys
 * @extends MODx.grid.Grid
 * @xtype i18n-grid-keys
 */

i18n.grid = i18n.grid || {};

i18n.grid.Keys = function (config) {
    config = config || {};

    // Selection model with checkboxes for multi-select
    var sm = new Ext.grid.CheckboxSelectionModel({
        singleSelect: false
    });

    Ext.applyIf(config, {
        id: 'i18n-grid-keys',

        url: i18n.config.connectorUrl,

        root: 'object.results',
        totalProperty: 'object.total',

        baseParams: {
            action: 'mgr/getlist',
            namespace: 'i18n'
        },

        fields: [
            'id',
            'key',
            'group',
            'description',
            'hash_default',
            'usage_count',
            'created_at',
            'updated_at',
            'last_used_at',
            'langs'
        ],

        paging: true,
        remoteSort: true,
        autoHeight: true,

        sm: sm,
        columns: [
            sm,
            {
                header: i18n.lex('i18n_key', 'Key'),
                dataIndex: 'key',
                sortable: true,
                width: 260
            },
            {
                header: i18n.lex('i18n_group', 'Group'),
                dataIndex: 'group',
                sortable: true,
                width: 130
            },
            {
                header: i18n.lex('i18n_description', 'Description'),
                dataIndex: 'description',
                sortable: false,
                width: 260
            },
            {
                header: i18n.lex('i18n_languages', 'Languages'),
                dataIndex: 'langs',
                sortable: false,
                width: 160,
                renderer: i18n.renderLangStatus
            },
            {
                header: i18n.lex('i18n_usage_count', 'Usage'),
                dataIndex: 'usage_count',
                sortable: true,
                width: 80
            },
            {
                header: i18n.lex('i18n_last_used_at', 'Last used at'),
                dataIndex: 'last_used_at',
                sortable: true,
                width: 150
            },
            {
                header: i18n.lex('i18n_updated_at', 'Updated'),
                dataIndex: 'updated_at',
                sortable: true,
                width: 150
            }
        ],

        tbar: [
            {
                id: 'i18n-btn-create',
                text: i18n.lex('i18n_create_key', 'Create key'),
                handler: function (btn, e) {
                    i18n.guardWrite(function () {
                        this.createKey(btn, e);
                    }, this);
                },
                scope: this,
                cls: 'primary-button'
            },
            '-',
            {
                id: 'i18n-btn-delete-selected',
                text: i18n.lex('i18n_delete_selected', 'Delete selected'),
                handler: function () {
                    i18n.guardWrite(function () {
                        this.deleteSelected();
                    }, this);
                },
                scope: this
            },
            '-',
            {
                id: 'i18n-btn-export',
                text: i18n.lex('i18n_export', 'Export'),
                handler: function () {
                    i18n.guardWrite(function () {
                        this.openExportWindow();
                    }, this);
                },
                scope: this
            },
            '-',
            {
                id: 'i18n-btn-rebuild-usage',
                text: i18n.lex('i18n_rebuild_usage', 'Rebuild usage'),
                handler: function () {
                    i18n.guardWrite(function () {
                        this.rebuildUsage();
                    }, this);
                },
                scope: this
            },
            '-',
            {
                xtype: 'xcheckbox',
                id: 'i18n-filter-only-unused',
                boxLabel: i18n.lex('i18n_only_unused', 'Only unused'),
                listeners: {
                    check: { fn: function (cb) { this.filterUnused(cb); }, scope: this }
                },
                cls: 'i18n-switch'
            },
            '->',
            {
                xtype: 'textfield',
                id: 'i18n-filter-search',
                emptyText: i18n.lex('i18n_search_key', 'Search key…'),
                listeners: {
                    change: { fn: function (tf) { this.search(tf); }, scope: this },
                    render: {
                        fn: function (tf) {
                            tf.getEl().addKeyListener(Ext.EventObject.ENTER, function () {
                                this.search(tf);
                            }, this);
                        },
                        scope: this
                    }
                },
                width: 260
            },
            '-',
            {
                xtype: 'textfield',
                id: 'i18n-filter-group',
                emptyText: i18n.lex('i18n_filter_group', 'Filter by group…'),
                listeners: {
                    change: { fn: function (tf) { this.filterGroup(tf); }, scope: this },
                    render: {
                        fn: function (tf) {
                            tf.getEl().addKeyListener(Ext.EventObject.ENTER, function () {
                                this.filterGroup(tf);
                            }, this);
                        },
                        scope: this
                    }
                },
                width: 160
            }
        ]
    });

    i18n.grid.Keys.superclass.constructor.call(this, config);

    // Initial state (based on current license)
    this.applyWriteState();

    // Keep toolbar gating in sync after license refresh/activation
    this.bindLicenseEvents();

    this.on('rowdblclick', this.onRowDblClick, this);
};

Ext.extend(i18n.grid.Keys, MODx.grid.Grid, {

    /**
     * Enable/disable actionable toolbar buttons based on current license.
     * Filters/search stay enabled.
     */
    applyWriteState: function () {
        var allow = (typeof i18n.isWriteAllowed === 'function') ? i18n.isWriteAllowed() : true;

        var ids = [
            'i18n-btn-create',
            'i18n-btn-delete-selected',
            'i18n-btn-export',
            'i18n-btn-rebuild-usage'
        ];

        for (var i = 0; i < ids.length; i++) {
            var cmp = Ext.getCmp(ids[i]);
            if (cmp && typeof cmp.setDisabled === 'function') {
                cmp.setDisabled(!allow);
            }
        }
    },

    /**
     * Subscribe to global license updates and keep the grid UI in sync.
     * Requires i18n.emitLicenseUpdated() from i18n.js (event: i18n:licenseUpdated).
     */
    bindLicenseEvents: function () {
        if (this._i18nLicenseBound) return;
        this._i18nLicenseBound = true;

        var grid = this;

        // If component is destroyed, stop updating it.
        var alive = function () {
            return grid && !grid.destroyed && grid.rendered !== false;
        };

        try {
            window.addEventListener('i18n:licenseUpdated', function () {
                if (!alive()) return;
                try {
                    grid.applyWriteState();
                } catch (e) {}
            });
        } catch (e) {}
    },

    onRowDblClick: function (grid, rowIndex) {
        var rec = this.getStore().getAt(rowIndex);
        if (!rec) return;
        this.showUsage(rec);
    },

    getMenu: function () {
        var m = [];
        var writeAllowed = (typeof i18n.isWriteAllowed === 'function') ? i18n.isWriteAllowed() : true;

        if (this.menu && this.menu.record) {

            // Copy key (always allowed)
            (function (grid) {
                function extractKey(rec) {
                    if (!rec) return '';
                    if (typeof rec.get === 'function') {
                        return (rec.get('key') || '').toString();
                    }
                    if (rec.data && typeof rec.data.key !== 'undefined') {
                        return (rec.data.key || '').toString();
                    }
                    if (typeof rec.key !== 'undefined') {
                        return (rec.key || '').toString();
                    }
                    return '';
                }

                var rec = grid.menu.record;
                var key = extractKey(rec);

                if (!key) {
                    var sm = grid.getSelectionModel && grid.getSelectionModel();
                    var sel = sm && sm.getSelected ? sm.getSelected() : null;
                    key = extractKey(sel);
                }

                function showManualCopy(text) {
                    MODx.msg.alert(
                        i18n.lex('i18n_copy_key', 'Copy key'),
                        '<textarea style="width:100%;height:60px" readonly="readonly">' +
                            Ext.util.Format.htmlEncode(text) +
                        '</textarea>' +
                        '<div style="margin-top:6px;opacity:.8">' +
                            Ext.util.Format.htmlEncode(i18n.lex('i18n_copy_hint', 'Press Ctrl+C to copy.')) +
                        '</div>'
                    );

                    setTimeout(function () {
                        var ta = document.querySelector('.x-window textarea');
                        if (ta) { ta.focus(); ta.select(); }
                    }, 60);
                }

                function tryCopy(text) {
                    try {
                        var ta = document.createElement('textarea');
                        ta.value = text;
                        ta.setAttribute('readonly', 'readonly');
                        ta.style.position = 'fixed';
                        ta.style.left = '-9999px';
                        ta.style.top = '0';
                        document.body.appendChild(ta);

                        ta.focus();
                        ta.select();

                        var ok = document.execCommand('copy');
                        document.body.removeChild(ta);
                        return !!ok;
                    } catch (e) {
                        return false;
                    }
                }

                m.push({
                    text: i18n.lex('i18n_copy_key', 'Copy key'),
                    handler: function () {
                        if (!key) {
                            MODx.msg.alert(
                                i18n.lex('error', 'Error'),
                                i18n.lex('i18n_copy_failed', 'Copy failed')
                            );
                            return;
                        }

                        if (navigator.clipboard && window.isSecureContext) {
                            navigator.clipboard.writeText(key).then(
                                function () {
                                    MODx.msg.status({
                                        title: i18n.lex('success', 'Success'),
                                        message: i18n.lex('i18n_copied', 'Copied')
                                    });
                                },
                                function () {
                                    if (tryCopy(key)) {
                                        MODx.msg.status({
                                            title: i18n.lex('success', 'Success'),
                                            message: i18n.lex('i18n_copied', 'Copied')
                                        });
                                    } else {
                                        showManualCopy(key);
                                    }
                                }
                            );
                            return;
                        }

                        if (tryCopy(key)) {
                            MODx.msg.status({
                                title: i18n.lex('success', 'Success'),
                                message: i18n.lex('i18n_copied', 'Copied')
                            });
                            return;
                        }

                        showManualCopy(key);
                    },
                    scope: grid
                });

                m.push('-');
            })(this);

            m.push({
                text: i18n.lex('i18n_usage_show', 'Show usage'),
                handler: function () {
                    this.showUsage(this.menu.record);
                },
                scope: this
            });

            // Write actions are hidden in read-only mode
            if (writeAllowed) {
                m.push('-');

                m.push({
                    text: i18n.lex('i18n_update_key', 'Edit key'),
                    handler: function () {
                        i18n.guardWrite(function () {
                            this.updateKey(this.menu.record);
                        }, this);
                    },
                    scope: this
                });

                m.push('-');

                m.push({
                    text: i18n.lex('i18n_delete_key', 'Delete key'),
                    handler: function () {
                        i18n.guardWrite(function () {
                            this.deleteKey();
                        }, this);
                    },
                    scope: this
                });

                m.push('-');

                m.push({
                    text: i18n.lex('i18n_rebuild_usage', 'Rebuild usage'),
                    handler: function () {
                        i18n.guardWrite(function () {
                            this.rebuildUsage();
                        }, this);
                    },
                    scope: this
                });
            } else {
                m.push('-');
                m.push({
                    text: i18n.lex('i18n_readonly_menu', 'Read-only mode'),
                    disabled: true
                });
            }
        }

        return m;
    },

    search: function (tf) {
        var store = this.getStore();
        store.baseParams.query = tf.getValue();
        this.getBottomToolbar().changePage(1);
        this.refresh();
    },

    filterGroup: function (tf) {
        var store = this.getStore();
        store.baseParams.group = tf.getValue();
        this.getBottomToolbar().changePage(1);
        this.refresh();
    },

    filterUnused: function (cb) {
        var store = this.getStore();
        store.baseParams.only_unused = cb.getValue() ? 1 : 0;
        this.getBottomToolbar().changePage(1);
        this.refresh();
    },

    rebuildUsage: function () {
        i18n.guardWrite(function () {
            MODx.msg.confirm({
                title: i18n.lex('i18n_rebuild_usage', 'Rebuild usage'),
                text: i18n.lex('i18n_rebuild_usage_confirm', 'Recalculate usage statistics?'),
                url: i18n.config.connectorUrl,
                params: {
                    action: 'mgr/rebuildusage',
                    namespace: i18n.config.namespace || 'i18n'
                },
                listeners: {
                    success: {
                        fn: function () {
                            this.refresh();
                        },
                        scope: this
                    }
                }
            });
        }, this);
    },

    createKey: function (btn, e) {
        i18n.guardWrite(function () {
            if (this.createKeyWindow) {
                this.createKeyWindow.close();
                this.createKeyWindow.destroy();
                this.createKeyWindow = null;
            }

            this.createKeyWindow = MODx.load({
                xtype: 'i18n-window-key-create',
                listeners: {
                    success: { fn: this.refresh, scope: this }
                }
            });

            this.createKeyWindow.setValues({});
            this.createKeyWindow.show(e && e.target ? e.target : btn.getEl());
        }, this);
    },

    updateKey: function () {
        i18n.guardWrite(function () {
            var rec = null;

            if (this.menu && this.menu.record) {
                rec = this.menu.record;
            }

            if (!rec) {
                var sm = this.getSelectionModel();
                if (sm && sm.getSelected) {
                    rec = sm.getSelected();
                }
            }

            if (!rec) {
                return;
            }

            var id = null;
            if (typeof rec.get === 'function') {
                id = rec.get('id');
            } else if (rec.data && typeof rec.data.id !== 'undefined') {
                id = rec.data.id;
            } else if (typeof rec.id !== 'undefined') {
                id = rec.id;
            }

            id = parseInt(id, 10);
            if (!id) {
                MODx.msg.alert(
                    i18n.lex('error', 'Error'),
                    i18n.lex('i18n_update_key_invalid_id', 'Missing or invalid key ID.')
                );
                return;
            }

            var grid = this;

            MODx.Ajax.request({
                url: i18n.config.connectorUrl,
                params: {
                    action: 'mgr/getkey',
                    namespace: i18n.config.namespace || 'i18n',
                    id: id
                },
                listeners: {
                    success: {
                        fn: function (r) {
                            var record = (r && r.object) ? r.object : null;

                            if (!record || !record.id) {
                                MODx.msg.alert(
                                    i18n.lex('error', 'Error'),
                                    i18n.lex('i18n_update_key_load_failed', 'Failed to load key data.')
                                );
                                return;
                            }

                            var w = MODx.load({
                                xtype: 'i18n-window-key-update',
                                record: record,
                                listeners: {
                                    success: { fn: grid.refresh, scope: grid }
                                }
                            });

                            if (w) {
                                w.show();
                            }
                        },
                        scope: this
                    },
                    failure: {
                        fn: function (r) {
                            MODx.msg.alert(
                                i18n.lex('error', 'Error'),
                                (r && r.message) ? r.message : i18n.lex('i18n_update_key_load_failed', 'Failed to load key data.')
                            );
                        },
                        scope: this
                    }
                }
            });
        }, this);
    },

    deleteKey: function () {
        i18n.guardWrite(function () {
            var rec = null;

            if (this.menu && this.menu.record) {
                rec = this.menu.record;
            }

            if (!rec) {
                var sm = this.getSelectionModel();
                if (sm && sm.getSelected) {
                    rec = sm.getSelected();
                }
            }

            if (!rec) {
                return;
            }

            var id = null;

            if (typeof rec.get === 'function') {
                id = rec.get('id');
            } else if (typeof rec.id !== 'undefined') {
                id = rec.id;
            } else if (rec.data && typeof rec.data.id !== 'undefined') {
                id = rec.data.id;
            }

            if (!id) {
                return;
            }

            MODx.msg.confirm({
                title: i18n.lex('i18n_delete_key', 'Delete key'),
                text: i18n.lex(
                    'i18n_delete_key_confirm',
                    'Delete this key and all its translations?'
                ),
                url: i18n.config.connectorUrl,
                params: {
                    action: 'mgr/delete',
                    namespace: i18n.config.namespace || 'i18n',
                    ids: id
                },
                listeners: {
                    success: {
                        fn: function () {
                            this.refresh();
                        },
                        scope: this
                    }
                }
            });
        }, this);
    },

    deleteSelected: function () {
        i18n.guardWrite(function () {
            var sm  = this.getSelectionModel();
            var sel = sm.getSelections ? sm.getSelections() : [];

            if (!sel || !sel.length) {
                MODx.msg.alert(
                    i18n.lex('i18n_delete_selected', 'Delete selected'),
                    i18n.lex('i18n_delete_selected_empty', 'No keys selected.')
                );
                return;
            }

            var ids = [];
            Ext.each(sel, function (rec) {
                var id = rec.get('id');
                if (id) {
                    ids.push(id);
                }
            });

            if (!ids.length) return;

            MODx.msg.confirm({
                title: i18n.lex('i18n_delete_selected', 'Delete selected'),
                text: i18n.lex(
                    'i18n_delete_selected_confirm',
                    'Delete all selected keys and their translations?'
                ),
                url: i18n.config.connectorUrl,
                params: {
                    action: 'mgr/delete',
                    namespace: i18n.config.namespace || 'i18n',
                    ids: ids.join(',')
                },
                listeners: {
                    success: {
                        fn: function () {
                            this.refresh();
                        },
                        scope: this
                    }
                }
            });
        }, this);
    },

    showUsage: function (rec) {
        var record = null;

        if (rec && typeof rec.get === 'function') {
            record = rec;
        } else if (this.menu && this.menu.record && typeof this.menu.record.get === 'function') {
            record = this.menu.record;
        } else {
            var sm = this.getSelectionModel();
            if (sm && sm.getSelected) {
                record = sm.getSelected();
            }
        }

        if (!record) {
            return;
        }

        var keyId   = record.get('id');
        var keyName = record.get('key') || '';

        if (!keyId) {
            return;
        }

        var win = MODx.load({
            xtype: 'i18n-window-usage',
            keyId: keyId,
            keyName: keyName
        });

        if (win) {
            win.show();
        }
    },

    openExportWindow: function () {
        i18n.guardWrite(function () {
            var sm        = this.getSelectionModel();
            var hasSelect = sm && sm.getCount && sm.getCount() > 0;

            MODx.Ajax.request({
                url: i18n.config.connectorUrl,
                params: {
                    action: 'mgr/getgroups',
                    namespace: i18n.config.namespace || 'i18n'
                },
                listeners: {
                    success: {
                        fn: function (r) {
                            var groups = [];

                            if (r.object && Ext.isArray(r.object.groups)) {
                                groups = r.object.groups;
                            } else if (Ext.isArray(r.groups)) {
                                groups = r.groups;
                            }

                            var win = MODx.load({
                                xtype: 'i18n-window-export',
                                grid: this,
                                availableGroups: groups,
                                hasSelection: hasSelect
                            });

                            if (win) {
                                win.show();
                            }
                        },
                        scope: this
                    },
                    failure: {
                        fn: function () {
                            var win = MODx.load({
                                xtype: 'i18n-window-export',
                                grid: this,
                                availableGroups: [],
                                hasSelection: hasSelect
                            });

                            if (win) {
                                win.show();
                            }
                        },
                        scope: this
                    }
                }
            });
        }, this);
    },

    exportKeys: function (options) {
        i18n.guardWrite(function () {
            options = options || {};

            var format = (options.format === 'csv') ? 'csv' : 'json';

            var params = {
                action: 'mgr/export',
                namespace: i18n.config.namespace || 'i18n',
                format: format
            };

            if (options.scope) {
                params.scope = options.scope;
            }
            if (options.ids) {
                params.ids = options.ids;
            }
            if (options.group) {
                params.group = options.group;
            }
            if (options.langs) {
                params.langs = options.langs;
            }

            MODx.Ajax.request({
                url: i18n.config.connectorUrl,
                params: params,
                listeners: {
                    success: {
                        fn: function (r) {
                            if (!r.object || !r.object.content) {
                                MODx.msg.alert(
                                    i18n.lex('i18n_export', 'Export'),
                                    i18n.lex('i18n_export_empty', 'Nothing to export.')
                                );
                                return;
                            }

                            var filename = r.object.filename || (
                                'i18n_export.' + (format === 'csv' ? 'csv' : 'json')
                            );
                            var mime = r.object.mime || (
                                format === 'csv'
                                    ? 'text/csv; charset=UTF-8'
                                    : 'application/json; charset=UTF-8'
                            );
                            var content = r.object.content;

                            try {
                                var blob = new Blob([content], { type: mime });

                                if (window.saveAs) {
                                    window.saveAs(blob, filename);
                                } else {
                                    var url = (window.URL || window.webkitURL).createObjectURL(blob);
                                    var a = document.createElement('a');
                                    a.href = url;
                                    a.download = filename;
                                    document.body.appendChild(a);
                                    a.click();
                                    setTimeout(function () {
                                        document.body.removeChild(a);
                                        (window.URL || window.webkitURL).revokeObjectURL(url);
                                    }, 100);
                                }
                            } catch (e) {
                                MODx.msg.alert(
                                    i18n.lex('i18n_export', 'Export'),
                                    i18n.lex(
                                        'i18n_export_download_error',
                                        'Export completed, but automatic download failed. Please copy the content from the browser console.'
                                    )
                                );
                                if (window.console && console.log) {
                                    console.log('i18n export [' + filename + ']:\n', content);
                                }
                            }
                        },
                        scope: this
                    },
                    failure: {
                        fn: function (r) {
                            MODx.msg.alert(
                                i18n.lex('error', 'Error'),
                                (r.message || i18n.lex('i18n_export_error', 'Error while exporting keys.'))
                            );
                        },
                        scope: this
                    }
                }
            });
        }, this);
    }
});

Ext.reg('i18n-grid-keys', i18n.grid.Keys);

/**
 * Renderer for language status column.
 */
i18n.renderLangStatus = function (value, meta, record) {
    var raw = value || record.data.langs || null;
    var list = [];

    if (raw && typeof raw === 'object' && !Ext.isArray(raw)) {
        Ext.iterate(raw, function (code, hasValue) {
            if (!hasValue) return;
            code = String(code || '').trim();
            if (!code) return;
            list.push(code);
        });
    } else if (Ext.isArray(raw)) {
        Ext.each(raw, function (code) {
            code = String(code || '').trim();
            if (!code) return;
            list.push(code);
        });
    } else if (typeof raw === 'string') {
        raw.split(',').forEach(function (code) {
            code = String(code || '').trim();
            if (!code) return;
            list.push(code);
        });
    }

    if (!list.length) {
        return '';
    }

    var items = [];

    Ext.each(list, function (code) {
        items.push(
            '<span class="i18n-lang has-value">' +
                Ext.util.Format.htmlEncode(code) +
            '</span>'
        );
    });

    return items.join('&nbsp;');
};
