/**
 * Settings form for i18n component (includes Licensing UI).
 *
 * Updated licensing UI:
 *  - Activation uses activation_code (mgr/license/activate)
 *  - Status is still loaded via mgr/license/status
 *  - Check-in uses mgr/license/checkin
 *
 * Notes:
 *  - Save is blocked in read-only mode (UI + server-side enforced via LicenseProcessor).
 *  - License activation must work even in read-only mode.
 */

i18n = window.i18n || {};
i18n.panel = i18n.panel || {};

i18n.panel.Settings = function (config) {
    config = config || {};

    Ext.applyIf(config, {
        id: 'i18n-panel-settings',
        layout: 'form',
        labelAlign: 'top',
        autoHeight: true,
        border: false,
        bodyStyle: 'padding: 15px',
        url: i18n.config.connectorUrl,
        baseParams: { action: 'mgr/settings/save' },
        defaults: {
            anchor: '100%',
            style: 'margin-bottom: 15px'
        },

        items: [
            // -------------------------
            // Settings
            // -------------------------
            {
                xtype: 'combo',
                name: 'default_lang',
                hiddenName: 'default_lang',
                fieldLabel: i18n.lex('i18n_default_lang', 'Default language'),
                mode: 'local',
                triggerAction: 'all',
                editable: true,
                forceSelection: false,
                anchor: '60%',

                store: new Ext.data.ArrayStore({
                    fields: ['code', 'label'],
                    data: [
                        ['en', i18n.lex('i18n_lang_en', 'English (en)')],
                        ['uk', i18n.lex('i18n_lang_uk', 'Ukrainian (uk)')],
                        ['ru', i18n.lex('i18n_lang_ru', 'Russian (ru)')],
                        ['pl', i18n.lex('i18n_lang_pl', 'Polish (pl)')],
                        ['de', i18n.lex('i18n_lang_de', 'German (de)')],
                        ['fr', i18n.lex('i18n_lang_fr', 'French (fr)')],
                        ['es', i18n.lex('i18n_lang_es', 'Spanish (es)')],
                        ['it', i18n.lex('i18n_lang_it', 'Italian (it)')],
                        ['pt', i18n.lex('i18n_lang_pt', 'Portuguese (pt)')],
                        ['nl', i18n.lex('i18n_lang_nl', 'Dutch (nl)')],
                        ['sv', i18n.lex('i18n_lang_sv', 'Swedish (sv)')],
                        ['no', i18n.lex('i18n_lang_no', 'Norwegian (no)')],
                        ['da', i18n.lex('i18n_lang_da', 'Danish (da)')],
                        ['fi', i18n.lex('i18n_lang_fi', 'Finnish (fi)')],
                        ['cs', i18n.lex('i18n_lang_cs', 'Czech (cs)')],
                        ['sk', i18n.lex('i18n_lang_sk', 'Slovak (sk)')],
                        ['ro', i18n.lex('i18n_lang_ro', 'Romanian (ro)')],
                        ['bg', i18n.lex('i18n_lang_bg', 'Bulgarian (bg)')],
                        ['sr', i18n.lex('i18n_lang_sr', 'Serbian (sr)')],
                        ['hr', i18n.lex('i18n_lang_hr', 'Croatian (hr)')],
                        ['lt', i18n.lex('i18n_lang_lt', 'Lithuanian (lt)')],
                        ['lv', i18n.lex('i18n_lang_lv', 'Latvian (lv)')],
                        ['et', i18n.lex('i18n_lang_et', 'Estonian (et)')],
                        ['hu', i18n.lex('i18n_lang_hu', 'Hungarian (hu)')],
                        ['tr', i18n.lex('i18n_lang_tr', 'Turkish (tr)')],
                        ['ar', i18n.lex('i18n_lang_ar', 'Arabic (ar)')],
                        ['zh', i18n.lex('i18n_lang_zh', 'Chinese Simplified (zh)')]
                    ]
                }),

                valueField: 'code',
                displayField: 'label',

                getValue: function () {
                    var v = Ext.form.ComboBox.prototype.getValue.call(this);
                    return (v || '').trim();
                },

                setValue: function (v) {
                    v = (v || '').trim();
                    Ext.form.ComboBox.prototype.setValue.call(this, v);
                }
            },
            {
                xtype: 'hidden',
                name: 'allowed_langs',
                itemId: 'allowed-langs-hidden'
            },
            {
                xtype: 'checkboxgroup',
                fieldLabel: i18n.lex('i18n_allowed_langs', 'Allowed languages'),
                itemId: 'allowed-langs-group',
                columns: 4,
                vertical: true,
                items: [
                    { boxLabel: i18n.lex('i18n_lang_en', 'English (en)'), name: 'allowed_langs_en', inputValue: 'en' },
                    { boxLabel: i18n.lex('i18n_lang_uk', 'Ukrainian (uk)'), name: 'allowed_langs_uk', inputValue: 'uk' },
                    { boxLabel: i18n.lex('i18n_lang_ru', 'Russian (ru)'), name: 'allowed_langs_ru', inputValue: 'ru' },
                    { boxLabel: i18n.lex('i18n_lang_pl', 'Polish (pl)'), name: 'allowed_langs_pl', inputValue: 'pl' },
                    { boxLabel: i18n.lex('i18n_lang_de', 'German (de)'), name: 'allowed_langs_de', inputValue: 'de' },
                    { boxLabel: i18n.lex('i18n_lang_fr', 'French (fr)'), name: 'allowed_langs_fr', inputValue: 'fr' },
                    { boxLabel: i18n.lex('i18n_lang_es', 'Spanish (es)'), name: 'allowed_langs_es', inputValue: 'es' },
                    { boxLabel: i18n.lex('i18n_lang_it', 'Italian (it)'), name: 'allowed_langs_it', inputValue: 'it' },
                    { boxLabel: i18n.lex('i18n_lang_pt', 'Portuguese (pt)'), name: 'allowed_langs_pt', inputValue: 'pt' },
                    { boxLabel: i18n.lex('i18n_lang_tr', 'Turkish (tr)'), name: 'allowed_langs_tr', inputValue: 'tr' },
                    { boxLabel: i18n.lex('i18n_lang_ar', 'Arabic (ar)'), name: 'allowed_langs_ar', inputValue: 'ar' },
                    { boxLabel: i18n.lex('i18n_lang_zh', 'Chinese Simplified (zh)'), name: 'allowed_langs_zh', inputValue: 'zh' }
                ]
            },
            {
                xtype: 'textfield',
                name: 'allowed_langs_custom',
                itemId: 'allowed-langs-custom',
                fieldLabel: i18n.lex('i18n_allowed_langs_custom', 'Custom codes CSV'),
                anchor: '100%'
            },
            {
                xtype: 'textfield',
                name: 'default_prefix',
                fieldLabel: i18n.lex('i18n_default_prefix', 'Default prefix'),
                anchor: '60%',
                listeners: {
                    change: function (f, v) {
                        f.setValue((v || '').trim());
                    }
                }
            },
            {
                xtype: 'textfield',
                name: 'cookie_name',
                fieldLabel: i18n.lex('i18n_cookie_name', 'Cookie name')
            },
            {
                xtype: 'numberfield',
                name: 'cookie_lifetime_days',
                fieldLabel: i18n.lex('i18n_cookie_lifetime_days', 'Cookie lifetime (days)'),
                allowDecimals: false,
                allowNegative: false,
                minValue: 1
            },
            {
                xtype: 'xcheckbox',
                name: 'mgr_tools_enabled',
                boxLabel: i18n.lex('i18n_mgr_tools_enabled', 'Enable floating i18n tools'),
                inputValue: 1
            },
            {
                xtype: 'xcheckbox',
                name: 'route_tools_enabled',
                boxLabel: i18n.lex('i18n_route_tools_enabled', 'Enable routing i18n tools'),
                inputValue: 1
            },
            {
                xtype: 'xcheckbox',
                name: 'track_usage',
                boxLabel: i18n.lex('i18n_track_usage', 'Track usage'),
                inputValue: 1
            },

            // -------------------------
            // Licensing
            // -------------------------
            {
                xtype: 'fieldset',
                title: i18n.lex('i18n_license', 'License'),
                autoHeight: true,
                collapsible: true,
                collapsed: false,
                items: [
                    {
                        xtype: 'panel',
                        id: 'i18n-license-status-panel',
                        border: false,
                        bodyStyle: 'padding: 10px; background: #fafafa; border: 1px solid #ddd;',
                        html: '<div>' + Ext.util.Format.htmlEncode(i18n.lex('i18n_loading', 'Loading...')) + '</div>'
                    },
                    {
                        xtype: 'textfield',
                        id: 'i18n-license-fingerprint',
                        fieldLabel: i18n.lex('i18n_license_fingerprint', 'Fingerprint'),
                        readOnly: true,
                        anchor: '100%'
                    },

                    // NEW: activation code (the only thing user receives from seller)
                    {
                        xtype: 'textfield',
                        id: 'i18n-activation-code',
                        fieldLabel: i18n.lex('i18n_activation_code', 'Activation code'),
                        anchor: '100%',
                        emptyText: 'I18N-XXXX-XXXX-XXXX-XXXX-XXXX',
                        listeners: {
                            change: function (f, v) {
                                f.setValue((v || '').trim().toUpperCase());
                            }
                        }
                    },

                    {
                        xtype: 'panel',
                        border: false,
                        bodyStyle: 'padding-top: 5px;',
                        items: [
                            {
                                xtype: 'button',
                                text: i18n.lex('i18n_license_refresh', 'Refresh status'),
                                handler: function () {
                                    var p = Ext.getCmp('i18n-panel-settings');
                                    if (p && p.refreshLicenseUi) p.refreshLicenseUi(true);
                                }
                            },
                            {
                                xtype: 'button',
                                text: i18n.lex('i18n_license_checkin', 'Check-in'),
                                style: 'margin-left: 10px;',
                                handler: function () {
                                    var p = Ext.getCmp('i18n-panel-settings');
                                    if (p && p.checkInLicenseFromUi) p.checkInLicenseFromUi();
                                }
                            },
                            {
                                xtype: 'button',
                                text: i18n.lex('i18n_license_activate', 'Activate'),
                                cls: 'primary-button',
                                style: 'margin-left: 10px;',
                                handler: function () {
                                    var p = Ext.getCmp('i18n-panel-settings');
                                    if (p && p.activateLicenseFromUi) p.activateLicenseFromUi();
                                }
                            }
                        ]
                    }
                ]
            }
        ],

        buttons: [
            {
                text: i18n.lex('save', 'Save'),
                cls: 'primary-button',
                handler: function () {
                    var panel = this;
                    if (typeof i18n.guardWrite === 'function') {
                        i18n.guardWrite(function () {
                            panel.submitSettings();
                        }, panel);
                    } else {
                        panel.submitSettings();
                    }
                },
                scope: this
            },
            {
                text: i18n.lex('reset', 'Reset'),
                handler: function () {
                    this.loadSettings();
                    this.refreshLicenseUi(false);
                },
                scope: this
            }
        ],

        listeners: {
            afterrender: {
                fn: function () {
                    this.loadSettings();
                    Ext.defer(function () { this.refreshLicenseUi(true); }, 50, this);
                },
                scope: this,
                single: true
            }
        }
    });

    i18n.panel.Settings.superclass.constructor.call(this, config);
};

Ext.extend(i18n.panel.Settings, MODx.FormPanel, {

    // -------------------------
    // Internal helpers (ExtJS collections are not stable across MODX builds)
    // -------------------------
    eachItemSafe: function (collection, fn, scope) {
        if (!collection) return;

        if (collection.each && typeof collection.each === 'function') {
            collection.each(fn, scope || this);
            return;
        }

        if (collection.items && Ext.isArray(collection.items)) {
            Ext.each(collection.items, fn, scope || this);
            return;
        }

        if (Ext.isArray(collection)) {
            Ext.each(collection, fn, scope || this);
            return;
        }
    },

    // -------------------------
    // Allowed langs helpers
    // -------------------------
    applyAllowedLangsToUi: function (csv) {
        var codes = [];
        if (typeof csv === 'string' && csv.length) {
            codes = csv.split(',').map(function (s) { return s.trim(); }).filter(function (s) { return !!s; });
        }

        var group = this.getComponent('allowed-langs-group');
        var customField = this.getComponent('allowed-langs-custom');

        if (group && group.items) {
            this.eachItemSafe(group.items, function (cb) {
                if (!cb || !cb.inputValue || typeof cb.setValue !== 'function') return;
                cb.setValue(codes.indexOf(cb.inputValue) !== -1);
            }, this);
        }

        if (customField && typeof customField.setValue === 'function') {
            var predefined = [];
            if (group && group.items) {
                this.eachItemSafe(group.items, function (cb) {
                    if (cb && cb.inputValue) predefined.push(cb.inputValue);
                }, this);
            }

            var custom = codes.filter(function (code) {
                return predefined.indexOf(code) === -1;
            });

            customField.setValue(custom.join(','));
        }
    },

    collectAllowedLangsCsv: function () {
        var codes = [];

        var group = this.getComponent('allowed-langs-group');
        var customField = this.getComponent('allowed-langs-custom');

        if (group && group.items) {
            this.eachItemSafe(group.items, function (cb) {
                if (!cb || typeof cb.getValue !== 'function') return;
                if (cb.getValue()) codes.push(cb.inputValue);
            }, this);
        }

        if (customField && typeof customField.getValue === 'function') {
            var extra = customField.getValue() || '';
            extra.split(',').forEach(function (raw) {
                var v = (raw || '').trim();
                if (!v) return;
                if (codes.indexOf(v) === -1) codes.push(v);
            });
        }

        return codes.join(',');
    },

    // -------------------------
    // Settings: Load / Save
    // -------------------------
    loadSettings: function () {
        var form = this.getForm();
        if (!form) return;

        MODx.Ajax.request({
            url: i18n.config.connectorUrl,
            params: {
                action: 'mgr/settings/get',
                namespace: i18n.config.namespace || 'i18n'
            },
            listeners: {
                success: {
                    fn: function (r) {
                        var data = r.object || {};
                        form.setValues(data);

                        Ext.defer(function () {
                            this.applyAllowedLangsToUi(data.allowed_langs || '');
                        }, 10, this);

                        try {
                            if (!i18n.config) i18n.config = {};
                            if (data.default_lang) i18n.config.defaultLang = data.default_lang;
                            if (typeof data.allowed_langs === 'string') i18n.config.allowedLangsRaw = data.allowed_langs;
                        } catch (e) {}
                    },
                    scope: this
                },
                failure: {
                    fn: function (r) {
                        MODx.msg.alert(
                            i18n.lex('error', 'Error'),
                            r.message || i18n.lex('i18n_settings_load_error', 'Error loading settings.')
                        );
                    },
                    scope: this
                }
            }
        });
    },

    submitSettings: function () {
        var form = this.getForm();
        if (!form || !form.isValid()) return;

        var csv = this.collectAllowedLangsCsv();
        var hidden = this.getComponent('allowed-langs-hidden');
        if (hidden && typeof hidden.setValue === 'function') hidden.setValue(csv);

        form.submit({
            url: i18n.config.connectorUrl,
            params: {
                action: 'mgr/settings/save',
                namespace: i18n.config.namespace || 'i18n'
            },
            waitMsg: i18n.lex('saving', 'Saving...'),
            success: function () {
                MODx.msg.status({ message: i18n.lex('i18n_settings_saved', 'Settings saved.'), delay: 3 });

                MODx.Ajax.request({
                    url: i18n.config.connectorUrl,
                    params: {
                        action: 'mgr/settings/get',
                        namespace: i18n.config.namespace || 'i18n'
                    },
                    listeners: {
                        success: {
                            fn: function (res) {
                                var obj = (res && res.object)
                                    ? res.object
                                    : (res && res.result && res.result.object)
                                        ? res.result.object
                                        : null;

                                if (obj) {
                                    try {
                                        var f = this.getForm && this.getForm();
                                        if (f && f.setValues) f.setValues(obj);
                                    } catch (e) {}

                                    Ext.defer(function () {
                                        this.applyAllowedLangsToUi(obj.allowed_langs || '');
                                    }, 10, this);
                                }

                                this.refreshLicenseUi(true, function (licStatus) {
                                    try {
                                        var detail = obj || {};
                                        detail.license = licStatus || null;
                                        window.dispatchEvent(new CustomEvent('i18n:settingsSaved', { detail: detail }));
                                    } catch (e) {}
                                }.createDelegate(this));
                            },
                            scope: this
                        },
                        failure: {
                            fn: function () { this.refreshLicenseUi(true); },
                            scope: this
                        }
                    }
                });
            },
            failure: function (f, r) {
                MODx.msg.alert(
                    i18n.lex('error', 'Error'),
                    (r && r.result && r.result.message) ? r.result.message : i18n.lex('i18n_settings_save_error', 'Error saving settings.')
                );
            },
            scope: this
        });
    },

    // -------------------------
    // Licensing UI
    // -------------------------
    normalizeLicenseStatus: function (st) {
        st = st || {};

        var mode = (st.mode !== undefined) ? String(st.mode) : 'readonly';

        var writeAllowed =
            (st.write_allowed === true || st.write_allowed === 1 || st.write_allowed === '1') ||
            (st.writeAllowed === true || st.writeAllowed === 1 || st.writeAllowed === '1');

        // New service returns ttl/grace_ttl (seconds)
        var trialLeft =
            (typeof st.trial_left_seconds === 'number') ? st.trial_left_seconds :
            (typeof st.trialLeftSeconds === 'number') ? st.trialLeftSeconds :
            (typeof st.ttl === 'number' && mode === 'trial') ? st.ttl : null;

        var graceLeft =
            (typeof st.grace_left_seconds === 'number') ? st.grace_left_seconds :
            (typeof st.graceLeftSeconds === 'number') ? st.graceLeftSeconds :
            (typeof st.grace_ttl === 'number' && mode === 'grace') ? st.grace_ttl : null;

        var fp = '';
        if (st.fingerprint !== undefined) fp = String(st.fingerprint || '');
        if (!fp && st.state && st.state.fingerprint) fp = String(st.state.fingerprint || '');

        var reason = (st.reason !== undefined) ? String(st.reason) : '';

        return {
            mode: mode,
            write_allowed: !!writeAllowed,
            reason: reason,
            fingerprint: fp,
            trial_left_seconds: trialLeft,
            grace_left_seconds: graceLeft
        };
    },

    getLicenseStatusFromConfig: function () {
        var lic = (i18n.config && i18n.config.license) ? i18n.config.license : {};
        return this.normalizeLicenseStatus(lic);
    },

    formatDuration: function (seconds) {
        seconds = parseInt(seconds, 10);
        if (!seconds || seconds <= 0) return '0';
        var d = Math.floor(seconds / 86400);
        var h = Math.floor((seconds % 86400) / 3600);
        var m = Math.floor((seconds % 3600) / 60);

        var parts = [];
        if (d) parts.push(d + 'd');
        if (h) parts.push(h + 'h');
        if (!d && m) parts.push(m + 'm');
        return parts.join(' ');
    },

    renderLicenseHtml: function (st) {
        st = this.normalizeLicenseStatus(st);

        var mode = st.mode || 'readonly';
        var writeAllowed = !!st.write_allowed;
        var reason = st.reason || '';
        var fp = st.fingerprint || '';

        var trialLeft = (typeof st.trial_left_seconds === 'number') ? st.trial_left_seconds : null;
        var graceLeft = (typeof st.grace_left_seconds === 'number') ? st.grace_left_seconds : null;

        var parts = [];
        parts.push('<div style="display:flex;gap:20px;flex-wrap:wrap">');

        parts.push('<div><b>' + Ext.util.Format.htmlEncode(i18n.lex('i18n_license_mode', 'Mode')) + ':</b> ' +
            Ext.util.Format.htmlEncode(mode) + '</div>');

        parts.push('<div><b>' + Ext.util.Format.htmlEncode(i18n.lex('i18n_write_allowed', 'Write allowed')) + ':</b> ' +
            Ext.util.Format.htmlEncode(writeAllowed ? 'yes' : 'no') + '</div>');

        if (reason) {
            parts.push('<div><b>' + Ext.util.Format.htmlEncode(i18n.lex('i18n_reason', 'Reason')) + ':</b> ' +
                Ext.util.Format.htmlEncode(reason) + '</div>');
        }

        parts.push('</div>');

        if (trialLeft !== null && mode === 'trial') {
            parts.push('<div style="margin-top:8px"><b>' + Ext.util.Format.htmlEncode(i18n.lex('i18n_trial_left', 'Trial left')) + ':</b> ' +
                Ext.util.Format.htmlEncode(this.formatDuration(trialLeft)) + '</div>');
        }

        if (graceLeft !== null && mode === 'grace') {
            parts.push('<div style="margin-top:8px"><b>' + Ext.util.Format.htmlEncode(i18n.lex('i18n_grace_left', 'Grace left')) + ':</b> ' +
                Ext.util.Format.htmlEncode(this.formatDuration(graceLeft)) + '</div>');
        }

        if (fp) {
            parts.push('<div style="margin-top:8px;opacity:.85"><b>' + Ext.util.Format.htmlEncode(i18n.lex('i18n_license_fingerprint', 'Fingerprint')) + ':</b> ' +
                Ext.util.Format.htmlEncode(fp) + '</div>');
        }

        if (mode === 'readonly') {
            parts.push(
                '<div style="margin-top:10px;padding:8px;border:1px solid #f0c36d;background:#fff8e1">' +
                Ext.util.Format.htmlEncode(i18n.lex(
                    'i18n_readonly_text',
                    'i18n is running in read-only mode. A valid license is required for write operations.'
                )) +
                '</div>'
            );
        } else if (mode === 'licensed-dev') {
            parts.push(
                '<div style="margin-top:10px;padding:8px;border:1px solid #c5e1a5;background:#f1f8e9">' +
                Ext.util.Format.htmlEncode(i18n.lex(
                    'i18n_license_dev_text',
                    'Development license is active. Deploy to a real domain and run Check-in to finalize.'
                )) +
                '</div>'
            );
        } else if (mode === 'grace') {
            parts.push(
                '<div style="margin-top:10px;padding:8px;border:1px solid #f0c36d;background:#fff8e1">' +
                Ext.util.Format.htmlEncode(i18n.lex(
                    'i18n_license_grace_text',
                    'License refresh is required. Please run Check-in to avoid switching to read-only mode.'
                )) +
                '</div>'
            );
        }

        return parts.join('');
    },

    refreshLicenseUi: function (forceServer, done) {
        var statusPanel = Ext.getCmp('i18n-license-status-panel');
        var fpField = Ext.getCmp('i18n-license-fingerprint');

        var applyStatus = function (stRaw) {
            var st = this.normalizeLicenseStatus(stRaw);

            if (statusPanel) {
                if (statusPanel.rendered && statusPanel.body) {
                    statusPanel.update(this.renderLicenseHtml(st));
                } else {
                    Ext.defer(function () { this.refreshLicenseUi(false, done); }, 30, this);
                    return;
                }
            }

            if (fpField && typeof fpField.setValue === 'function') {
                fpField.setValue(st.fingerprint || '');
            }

            try {
                if (!i18n.config) i18n.config = {};
                i18n.config.license = stRaw || st;
            } catch (e) {}

            if (typeof done === 'function') done(st);
        }.createDelegate(this);

        if (forceServer && typeof i18n.refreshLicenseStatus === 'function') {
            i18n.refreshLicenseStatus(function (ok, r) {
                var obj = (typeof i18n._extractObject === 'function')
                    ? i18n._extractObject(r)
                    : (r && r.object ? r.object : null);
                applyStatus(obj || this.getLicenseStatusFromConfig());
            }.createDelegate(this));
            return;
        }

        applyStatus(this.getLicenseStatusFromConfig());
    },

    checkInLicenseFromUi: function () {
        if (typeof i18n.checkInLicense !== 'function') {
            MODx.msg.alert(i18n.lex('error', 'Error'),
                i18n.lex('i18n_checkin_missing', 'Check-in is not available (client helper missing).'));
            return;
        }

        MODx.msg.status({ message: i18n.lex('i18n_checkin_progress', 'Checking in...'), delay: 2 });

        i18n.checkInLicense(function (ok, r) {
            if (ok) {
                MODx.msg.status({ message: i18n.lex('i18n_checkin_ok', 'Check-in completed.'), delay: 3 });
            } else {
                MODx.msg.alert(
                    i18n.lex('error', 'Error'),
                    (r && r.message) ? r.message :
                    (r && r.result && r.result.message) ? r.result.message :
                    i18n.lex('i18n_checkin_failed', 'Check-in failed.')
                );
            }
            this.refreshLicenseUi(true);
        }.createDelegate(this));
    },

    /**
     * NEW activation flow:
     * - user enters activation_code
     * - POST to mgr/license/activate with activation_code
     */
    activateLicenseFromUi: function () {
        var codeField = Ext.getCmp('i18n-activation-code');
        var code = codeField ? String(codeField.getValue() || '').trim() : '';
        code = code.toUpperCase();

        if (!code) {
            MODx.msg.alert(i18n.lex('error', 'Error'),
                i18n.lex('i18n_activation_code_missing', 'Activation code is required.'));
            return;
        }

        MODx.Ajax.request({
            url: i18n.config.connectorUrl,
            params: {
                action: 'mgr/license/activate',
                namespace: i18n.config.namespace || 'i18n',
                activation_code: code
            },
            listeners: {
                success: {
                    fn: function () {
                        MODx.msg.status({ message: i18n.lex('i18n_license_activated', 'License activated.'), delay: 3 });
                        this.refreshLicenseUi(true);
                    },
                    scope: this
                },
                failure: {
                    fn: function (r) {
                        MODx.msg.alert(
                            i18n.lex('error', 'Error'),
                            (r && r.message) ? r.message :
                            (r && r.result && r.result.message) ? r.result.message :
                            i18n.lex('i18n_license_activate_failed', 'License activation failed.')
                        );
                        this.refreshLicenseUi(true);
                    },
                    scope: this
                }
            }
        });
    }

});

Ext.reg('i18n-panel-settings', i18n.panel.Settings);
