/**
 * @class i18n.window.Usage
 * @extends MODx.Window
 * @xtype i18n-window-usage
 */

i18n.window = i18n.window || {};

i18n.window.Usage = function (config) {
    config = config || {};

    Ext.applyIf(config, {
        id: 'i18n-window-usage',
        title: i18n.lex('i18n_usage_title', 'Key usage') +
            (config.keyName ? ': ' + Ext.util.Format.htmlEncode(config.keyName) : ''),
        width: 800,
        height: 400,
        minHeight: 400,
        maxHeight: 600,
        autoHeight: false,
        closeAction: 'close',
        modal: true,
        layout: 'fit',
        resizable: true,
        maximizable: true,
        constrain: true,
        constrainHeader: true,
        keyId: config.keyId || null,
        keyName: config.keyName || '',
        key: config.key || '',
        items: [{
            xtype: 'panel',
            border: false,
            bodyCssClass: 'main-wrapper',
            autoScroll: true,
            html: '<p>' + i18n.lex('i18n_usage_loading', 'Loading usage data…') + '</p>'
        }],
        buttons: [{
            text: i18n.lex('close', 'Close'),
            handler: function () {
                this.close();
            },
            scope: this
        }]
    });

    i18n.window.Usage.superclass.constructor.call(this, config);

    this.on('show', this.loadUsage, this, { single: true });
};

Ext.extend(i18n.window.Usage, MODx.Window, {

    loadUsage: function () {
        var keyId = this.keyId;

        if (!keyId) {
            this.updateBodyHtml(
                '<p>' + i18n.lex('i18n_usage_no_key', 'No key ID provided.') + '</p>'
            );
            return;
        }

        var win = this;

        MODx.Ajax.request({
            url: i18n.config.connectorUrl,
            params: {
                action: 'mgr/getusage',
                namespace: i18n.config.namespace || 'i18n',
                key_id: keyId,
                key: this.key || ''
            },
            listeners: {
                success: {
                    fn: function (r) {
                        var results = [];
                        var total = 0;

                        if (r.object) {
                            if (Ext.isArray(r.object.results)) {
                                results = r.object.results;
                            }
                            if (typeof r.object.total === 'number') {
                                total = r.object.total;
                            }
                        }

                        if (!results.length && Ext.isArray(r.results)) {
                            results = r.results;
                        }
                        if (!total && typeof r.total === 'number') {
                            total = r.total;
                        }

                        if (!results.length) {
                            win.updateBodyHtml(
                                '<p>' + i18n.lex(
                                    'i18n_usage_empty',
                                    'No usage entries found for this key.'
                                ) + '</p>'
                            );
                            return;
                        }

                        var html = win.buildUsageTable(results, total || results.length);
                        win.updateBodyHtml(html);
                    },
                    scope: this
                },
                failure: {
                    fn: function (r) {
                        var msg = i18n.lex(
                            'i18n_usage_error',
                            'Error while loading usage data.'
                        );
                        if (r.message) {
                            msg += '<br><br>' + Ext.util.Format.htmlEncode(r.message);
                        }
                        win.updateBodyHtml('<p>' + msg + '</p>');
                    },
                    scope: this
                }
            }
        });
    },

    buildUsageTable: function (rows, total) {
        var thType = i18n.lex('i18n_usage_context_type', 'Context type');
        var thName = i18n.lex('i18n_usage_context_name', 'Context name');
        var thField = i18n.lex('i18n_usage_field', 'Field');
        var thId = i18n.lex('i18n_usage_context_id', 'Context ID');
        var thPrev = i18n.lex('i18n_usage_preview', 'Preview');
        var thDate = i18n.lex('i18n_usage_last_scan_at', 'Last scan at');

        var managerUrl = MODx.config.manager_url || '/manager/';

        var out = [];
        out.push('<p>' + i18n.lex('i18n_usage_total', 'Total usage:') + ' ' + total + '</p>');
        out.push('<table class="grid" style="width:100%; border-collapse:collapse;">');
        out.push('<thead><tr>');
        out.push('<th>' + Ext.util.Format.htmlEncode(thType) + '</th>');
        out.push('<th>' + Ext.util.Format.htmlEncode(thName) + '</th>');
        out.push('<th>' + Ext.util.Format.htmlEncode(thField) + '</th>');
        out.push('<th>' + Ext.util.Format.htmlEncode(thId) + '</th>');
        out.push('<th>' + Ext.util.Format.htmlEncode(thPrev) + '</th>');
        out.push('<th>' + Ext.util.Format.htmlEncode(thDate) + '</th>');
        out.push('</tr></thead>');
        out.push('<tbody>');

        Ext.each(rows, function (row) {
            var ctxType = row.context_type || '';
            var ctxName = row.context_name || '';
            var field = row.field || '';
            var ctxId = row.context_id !== null ? String(row.context_id) : '';
            var preview = row.preview || '';
            var lastScanAt = row.last_scan_at || '';

            var safeType = Ext.util.Format.htmlEncode(ctxType);
            var safeField = Ext.util.Format.htmlEncode(field);
            var safePrev = Ext.util.Format.htmlEncode(preview);
            var safeDate = Ext.util.Format.htmlEncode(lastScanAt);

            var displayName = ctxName || (ctxType + ' #' + ctxId);
            var safeName = Ext.util.Format.htmlEncode(displayName);
            var linkId = ctxId;
            var safeId = Ext.util.Format.htmlEncode(linkId);

            var linkHtml = safeName;
            if (ctxType && linkId) {
                var action = null;

                switch (ctxType) {
                    case 'resource':
                        action = 'resource/update';
                        break;
                    case 'template':
                        action = 'element/template/update';
                        break;
                    case 'chunk':
                        action = 'element/chunk/update';
                        break;
                    case 'snippet':
                        action = 'element/snippet/update';
                        break;
                    case 'plugin':
                        action = 'element/plugin/update';
                        break;
                    case 'tv_value':
                        action = 'resource/update';
                        break;
                    default:
                        action = null;
                }

                if (action) {
                    var url = managerUrl + '?a=' +
                        encodeURIComponent(action) +
                        '&id=' +
                        encodeURIComponent(linkId);

                    linkHtml =
                        '<a href="' + url + '" target="_blank" rel="noopener noreferrer">' +
                        safeName +
                        '</a>';
                }
            }

            out.push('<tr>');
            out.push('<td>' + safeType + '</td>');
            out.push('<td>' + linkHtml + '</td>');
            out.push('<td>' + safeField + '</td>');
            out.push('<td>' + safeId + '</td>');
            out.push('<td>' + safePrev + '</td>');
            out.push('<td>' + safeDate + '</td>');
            out.push('</tr>');
        });

        out.push('</tbody></table>');

        return out.join('');
    },

    updateBodyHtml: function (html) {
        var panel = this.items.get(0);
        if (panel && panel.body) {
            panel.body.update(html);
        }
    }
});

Ext.reg('i18n-window-usage', i18n.window.Usage);
